<?php

namespace onespace\widgets;

use Yii;
use yii\base\Widget;
use yii\helpers\Json;

/**
 * This widget is used to generate Apex Charts on the fly in PHP.
 * 
 * A chart should be called as follows:
 * 
 * ```php
 * echo ApexCharts::widget( [
 *  'options' => [
 *      ...
 *  ],
 * ] );
 * ```
 * 
 * To fill in the options, please refer to the Apex Charts documentation, and conform
 * your options to as an assosiative array matching the object parsed to the Javascript
 * renderer.
 * 
 * @see https://apexcharts.com/docs/
 * @see https://apexcharts.com/javascript-chart-demos/
 * 
 * @author  Gareth Palmer <gareth@one-space.co.za>
 */

class ApexCharts extends Widget {

    /**
     * Link to the CDN for Apex Charts, if needed.
     * 
     * @var string  CDN
     */

    const CDN = '<script src="https://cdn.jsdelivr.net/npm/apexcharts"></script>';

    /**
     * The parsed options used by the widget and ultimately rendered
     * into a chart.
     * 
     * @var array   $options
     * 
     * @access  public
     */

    public array $options;


    /**
     * @inheritdoc
     */

    public function run(): string {
        parent::run();
        $id = str_replace('-', '_', Yii::$app->security->generateRandomString(8));
        $options = Json::encode($this->options);
        $script = "<script defer>const chart{$id} = new ApexCharts(document.getElementById('$id'), {$options}); chart{$id}.render();</script>";
        return "<div id='{$id}'></div>{$script}";
    }


    /**
     * Returns, or if desired, directly echos the CDN for Apex Charts.
     * 
     * @param   bool    $echo   Default: false
     * 
     * @return  string
     * 
     * @static
     * @access  public
     * 
     * @see https://apexcharts.com/docs/installation/
     */

    public static function loadCDN(bool $echo = false): string {
        if ($echo) {
            echo self::CDN;
        }
        return self::CDN;
    }
}
