<?php

namespace common\components;

use Exception;
use libphonenumber\NumberParseException;
use libphonenumber\PhoneNumberFormat;
use libphonenumber\PhoneNumberType;
use libphonenumber\PhoneNumberUtil;
use yii\helpers\Inflector;
use yii\validators\Validator;

/**
 * Phone validator class that validates phone numbers for given
 * country and formats.
 * Country codes and attributes value should be ISO 3166-1 alpha-2 codes
 * @property string $countryAttribute The country code attribute of model
 * @property string $country The country is fixed
 * @property int|array $type The type of number it should be - can be left empty, or an array
 * @property bool $strict If country is not set or selected adds error
 * @property bool $format If phone number is valid formats value with
 * @property Closure $formatter A method to complete final formatting of the string. Should accept a string, and must return a string
 *          libphonenumber/PhoneNumberFormat const (default to INTERNATIONAL)
 */
class PhoneValidator extends Validator {
    public $strict = true;
    public $countryAttribute;
    public $country;
    public $type;
    public $formatter;

    public $format = true;
    public $includePlus = false;

    public function validateAttribute($model, $attribute) {
        
        if ($this->format === true) {
            $this->format = PhoneNumberFormat::INTERNATIONAL;
        }
        
        // if countryAttribute is set
        if (!isset($country) && isset($this->countryAttribute)) {
            $countryAttribute = $this->countryAttribute;
            $country = $model->$countryAttribute;
        }

        // if country is fixed
        if (!isset($country) && isset($this->country)) {
            $country = $this->country;
        }

        // if none select from our models with best effort
        if (!isset($country) && isset($model->country_code)) {
            $country = $model->country_code;
        }

        if (!isset($country) && isset($model->country)) {
            $country = $model->country;
        }

        // if none and strict
        if (!isset($country) && $this->strict) {
            $this->addError($model, $attribute, \Yii::t('udokmeci.phone.validator', 'For phone validation country required'));
            return false;
        }

        if (!isset($country)) {
            return true;
        }

        $phoneUtil = PhoneNumberUtil::getInstance();
        try {
            // add a plus if needed
            $number = $model->$attribute;
            if ($number[0] != '0' && $number[0] != '+') {
                $number = '+' . $number;
            }
            $numberProto = $phoneUtil->parse($number, $country);
            if ($phoneUtil->isValidNumber($numberProto)) {
                if (isset($this->type)) {
                    $type = $phoneUtil->getNumberType($numberProto);
                    if (is_array($this->type)) {
                        if (!in_array($type, $this->type)) {
                            $this->addError($model, $attribute, $model->getAttributeLabel($attribute) . ' must be a ' . Inflector::sentence(self::getTypeStrings($this->type), ' or ') . ' number');
                            return false;
                        }
                    } elseif ($type != $this->type) {
                        $this->addError($model, $attribute, $model->getAttributeLabel($attribute) . ' must be a ' . self::getTypeString($this->type) . ' number');
                        return false;
                    }
                }
                if (is_numeric($this->format)) {
                    $model->$attribute = $phoneUtil->format($numberProto, $this->format);
                }
                if (!$this->includePlus) {
                    $model->$attribute = str_replace('+', '', $model->$attribute);
                }
                if ($this->formatter != null) {
                    $model->$attribute = $this->formatter->call($this, $model->$attribute);
                }
                return true;
            } else {
                $this->addError($model, $attribute, $model->getAttributeLabel($attribute) . ' does not seem to be a valid phone number');
                return false;
            }
        } catch (NumberParseException $e) {
            $this->addError($model, $attribute, 'Unexpected Phone Number Format');
        } catch (Exception $e) {
            $this->addError($model, $attribute, 'Unexpected Phone Number Format or Country Code');
        }
    }

    public static function getTypeString($value) {
        return self::typeStrings()[$value];
    }

    public static function getTypeStrings($values) {
        $labels = [];
        foreach ($values as $value) {
            $labels[] = self::typeStrings()[$value];
        }
        return $labels;
    }

    public static function typeStrings() {
        return [
            PhoneNumberType::FIXED_LINE => 'Landline',
            PhoneNumberType::MOBILE => 'Mobile',
            PhoneNumberType::FIXED_LINE_OR_MOBILE => 'Landline/Mobile',
            PhoneNumberType::TOLL_FREE => 'Toll Free',
            PhoneNumberType::PREMIUM_RATE => 'Premium Rate',
            PhoneNumberType::SHARED_COST => 'Shared Cost',
            PhoneNumberType::VOIP => 'VOIP',
            PhoneNumberType::PERSONAL_NUMBER => 'Personal Number',
            PhoneNumberType::PAGER => 'Pager',
            PhoneNumberType::UAN => 'UAN',
            PhoneNumberType::UNKNOWN => 'Unknown',
            PhoneNumberType::EMERGENCY => 'Emergency',
            PhoneNumberType::VOICEMAIL => 'Voicemail',
            PhoneNumberType::SHORT_CODE => 'Short Code',
            PhoneNumberType::STANDARD_RATE => 'Standard Rate',
        ];
    }
}
