<?php

namespace onespace\tools\widgets\ui\panels;

use onespace\tools\widgets\ui\base\PanelBase;

/**
 * Draws out the sub page element the "Duel Panel" Which is a double column panel into which
 * the contents of the page should be rendered. Handles the styling etc automatically.
 * 
 * @author  Gareth Palmer <gareth@one-space.co.za>
 */

class DuelPanel extends PanelBase {

    /**
     * The contents of the left hand panel. Parse html, widgets, $this->render or even just a plain string.
     * 
     * @var string|array  $leftPanel
     * 
     * @access  public
     */

    public string|array $leftPanel;

    /**
     * The contents of the right hand panel. Parse html, widgets, $this->render or even just a plain string.
     * 
     * @var string|array  $rightPanel
     * 
     * @access  public
     */

    public string|array $rightPanel;

    /**
     * If desired place a banner on the top of the left hand panel. Parse html, widgets, $this->render or even just a plain string.
     * 
     * Designed to work with the `TopBanner` widget.
     * 
     * @var string  $bannerLeft
     * 
     * @access  public
     */

    public string $bannerLeft;

    /**
     * If desired place a banner on the top of the right hand panel. Parse html, widgets, $this->render or even just a plain string.
     * 
     * Designed to work with the `TopBanner` widget.
     * 
     * @var string  $bannerRight
     * 
     * @access  public
     */

    public string $bannerRight;

    /**
     * Add a custom class to the left panel.
     * 
     * @var string  $leftPanelClass
     * 
     * @access  public
     */

    public string $leftPanelClass;

    /**
     * Add a custom class to the right panel.
     * 
     * @var string  $rightPanelClass
     * 
     * @access  public
     */

    public string $rightPanelClass;


    /**
     * Whether or not the duel panels are resizable.
     * 
     * @var bool    $resizable     Default: false
     * 
     * @access  public
     */

    public bool $resizable = false;

    /**
     * Set a specific size configuration for the panels.
     * 
     * @see src/ui/panels/PanelSizes.php
     * 
     * @var PanelSizes  $panelSize
     * 
     * @access  public
     */

    public PanelSizes $panelSize;


    /**
     * {@inheritdoc}
     * 
     * @access  public
     */

    public function init(): void {
        parent::init();

        $this->panelSize ??= PanelSizes::FIFTY_FIFTY;

        $this->leftPanelClass ??= '';
        $this->rightPanelClass ??= '';

        if ($this->leftPanelClass !== '') {
            $this->leftPanelClass = ' ' . $this->leftPanelClass;
        }
        if ($this->rightPanelClass !== '') {
            $this->rightPanelClass = ' ' . $this->rightPanelClass;
        }
    }


    /**
     * {@inheritdoc}
     * 
     * @return  string
     */

    public function run(): string {
        $panel = parent::run();

        $main_class = 'ui-panel-duel';
        if ($this->resizable) {
            $main_class .= ' ui-panel-duel-resizable';
        }
        $main_class .= $this->panelSize->panelClass($this->resizable);
        $panel .= $this->set_main_div($main_class);

        $panel .= '<div class="ui-panel ui-pane-container ui-pane-container-left">';
        if (isset($this->bannerLeft)) {
            $panel .= $this->bannerLeft;
        }
        $panel .= "<div class='ui-pane-content{$this->leftPanelClass}'>";
        if (isset($this->leftPanel)) {
            if (is_array($this->leftPanel)) {
                $panel .= implode('', $this->leftPanel);
            } else {
                $panel .= $this->leftPanel;
            }
        }
        $panel .= "</div>"; // ui-pane-content
        $panel .= "</div>";

        if ($this->resizable) {
            $panel .= "<div class='ui-duel-pane-resizer' data-resizer></div>";
        }

        $panel .= '<div class="ui-panel ui-pane-container ui-pane-container-right">';
        if (isset($this->bannerRight)) {
            $panel .= $this->bannerRight;
        }
        $panel .= "<div class='ui-pane-content{$this->rightPanelClass}'>";
        if (isset($this->rightPanel)) {
            if (is_array($this->rightPanel)) {
                $panel .= implode('', $this->rightPanel);
            } else {
                $panel .= $this->rightPanel;
            }
        }
        $panel .= "</div>"; // ui-pane-content
        $panel .= "</div>";

        $panel .= "</div>"; // ui-panel

        return $panel;
    }
}
