<?php

namespace onespace\tools\widgets\ui\panels;

use onespace\tools\widgets\ui\base\PanelBase;

/**
 * Draws out the sub page element the "Duel Panel" Which is a double column panel into which
 * the contents of the page should be rendered. Handles the styling etc automatically.
 * 
 * @author  Gareth Palmer <gareth@one-space.co.za>
 */

class DuelPanel extends PanelBase {

    /**
     * The contents of the left hand panel. Parse html, widgets, $this->render or even just a plain string.
     * 
     * @var string|array  $left_panel
     * 
     * @access  public
     */

    public string|array $left_panel;

    /**
     * The contents of the right hand panel. Parse html, widgets, $this->render or even just a plain string.
     * 
     * @var string|array  $right_panel
     * 
     * @access  public
     */

    public string|array $right_panel;

    /**
     * If desired place a banner on the top of the left hand panel. Parse html, widgets, $this->render or even just a plain string.
     * 
     * Designed to work with the `TopBanner` widget.
     * 
     * @var string  $banner_left
     * 
     * @access  public
     */

    public string $banner_left;

    /**
     * If desired place a banner on the top of the right hand panel. Parse html, widgets, $this->render or even just a plain string.
     * 
     * Designed to work with the `TopBanner` widget.
     * 
     * @var string  $banner_right
     * 
     * @access  public
     */

    public string $banner_right;

    /**
     * Add a custom class to the left panel.
     * 
     * @var string  $left_panel_class
     * 
     * @access  public
     */

    public string $left_panel_class;

    /**
     * Add a custom class to the left panel.
     * 
     * @var string  $right_panel_class
     * 
     * @access  public
     */

    public string $right_panel_class;


    /**
     * Whether or not the duel panels are resizable
     * 
     * @var bool    $resizable     Default: false
     * 
     * @access  public
     */

    public bool $resizable = false;

    /**
     * Set a specific size configuration for the panels
     * 
     * @see src/ui/panels/PanelSizes.php
     * 
     * @var PanelSizes  $panelSize
     * 
     * @access  public
     */

    public PanelSizes $panelSize;


    /**
     * {@inheritdoc}
     * 
     * @access  public
     */

    public function init(): void {
        parent::init();

        $this->panelSize ??= PanelSizes::FIFTY_FIFTY;

        $this->left_panel_class ??= '';
        $this->right_panel_class ??= '';

        if ($this->left_panel_class !== '') {
            $this->left_panel_class = ' ' . $this->left_panel_class;
        }
        if ($this->right_panel_class !== '') {
            $this->right_panel_class = ' ' . $this->right_panel_class;
        }
    }


    /**
     * {@inheritdoc}
     * 
     * @return  string
     */

    public function run(): string {
        $panel = parent::run();

        $main_class = 'ui-panel-duel';
        if ($this->resizable) {
            $main_class .= ' ui-panel-duel-resizable';
        }
        $main_class .= $this->panelSize->panelClass($this->resizable);
        $panel .= $this->set_main_div($main_class);

        $panel .= '<div class="ui-panel ui-pane-container ui-pane-container-left">';
        if (isset($this->banner_left)) {
            $panel .= $this->banner_left;
        }
        $panel .= "<div class='ui-pane-content{$this->left_panel_class}'>";
        if (isset($this->left_panel)) {
            if (is_array($this->left_panel)) {
                $panel .= implode('', $this->left_panel);
            } else {
                $panel .= $this->left_panel;
            }
        }
        $panel .= "</div>"; // ui-pane-content
        $panel .= "</div>";

        if ($this->resizable) {
            $panel .= "<div class='ui-duel-pane-resizer' data-resizer></div>";
        }

        $panel .= '<div class="ui-panel ui-pane-container ui-pane-container-right">';
        if (isset($this->banner_right)) {
            $panel .= $this->banner_right;
        }
        $panel .= "<div class='ui-pane-content{$this->right_panel_class}'>";
        if (isset($this->right_panel)) {
            if (is_array($this->right_panel)) {
                $panel .= implode('', $this->right_panel);
            } else {
                $panel .= $this->right_panel;
            }
        }
        $panel .= "</div>"; // ui-pane-content
        $panel .= "</div>";

        $panel .= "</div>"; // ui-panel

        return $panel;
    }
}
