<?php

namespace onespace\widgets\grapeJS;

use Exception;
use onespace\widgets\ui\base\Tools;
use yii\base\Widget;
use yii\helpers\Html;
use Yii;

/**
 * Load up the GrapesJS editor.
 * 
 * {@inheritdoc}
 * 
 * @see https://grapesjs.com/docs
 * @see https://grapesjs.com/docs/getting-started.html
 * 
 * @author  Gareth Palmer <gareth@one-space.co.za>
 */

final class GrapesJS extends Widget {

    use Tools;

    public const DONT_STORE = 0;
    public const STORE_LOCAL = 1;
    public const STORE_REMOTE = 2;
    public const STORE_LOCAL_INLINE = 3;

    /**
     * OPTIONAL. Whether or not to use the newsletter plugin.
     * 
     * @var bool    $isNewsletter   Default: false
     * 
     * @access  public
     */

    public bool $isNewsletter = false;

    /**
     * OPTIONAL. The height of editor.
     * 
     * @var string  $height Default: 1000px
     * 
     * @access  public
     */

    public string $height = '1000px';

    /**
     * OPTIONAL. The height of editor.
     * 
     * @var string  $width  Default: auto
     * 
     * @access  public
     */

    public string $width = 'auto';

    /**
     * The content of the div to preload.
     * 
     * @var string  $content
     * 
     * @access  public
     */

    public string $content = '';

    /**
     * OPTIONAL. Whether or not to load the contents directly from the contents of the div.
     * 
     * @var bool    $fromElement    Default: true
     * 
     * @access  public
     */

    public bool $fromElement = true;

    /**
     * OPTIONAL. A completely custom loading script. If set, this will overwrite all other settings.
     * 
     * @var string  $customLoadScript
     * 
     * @access  public
     */

    public string $customLoadScript;

    /**
     * OPTIONAL. Any extra scripting options to add to the page when loading.
     * 
     * @var string  $extraScripts
     * 
     * @access  public
     */

    public string $extraScripts;

    /**
     * OPTIONAL. A completely custom styles sheet. If set, this will overwrite all other styles.
     * 
     * @var string  $customLoadScript
     * 
     * @access  public
     */

    public string $customStyles;

    /**
     * OPTIONAL. The type of storage used by the system.
     * 
     * @var int $storageType    Default: `self::DONT_STORE`
     * 
     * @access  public
     */

    public int $storageType;

    /**
     * OPTIONAL. The storage key needed for local storage.
     * 
     * @var string  $storageKey
     * 
     * @access  public
     */

    public string $storageKey;

    /**
     * OPTIONAL. Any HTML elements to add to the main DIV.
     * 
     * @var array   $options    Default: []
     * 
     * @access  public
     */

    public array $options = [];

    /**
     * OPTIONAL. The ID used throughout the widget to identify this specific instance.
     * This allows for multiple instances of the widget on the same page.
     * 
     * @var string  $id
     * 
     * @access  public
     */

    public string $id;

    /**
     * Whether to load the widget in debug mode or not.
     * 
     * @var bool    $debug
     * 
     * @access  public
     */

    public bool $debug = false;


    /**
     * {@inheritdoc}
     * 
     * @throws  Exception   When not storage key is defined and storing locally.
     * 
     * @access  public
     */

    public function init(): void {
        $this->id ??= str_replace('-', '_', 'gjs' . Yii::$app->security->generateRandomString(12));

        $this->options['id'] = $this->id;
        $this->options['class'] ??= 'grapes-js';

        $this->extraScripts ??= '';

        $this->storageType ??= self::DONT_STORE;
        if ($this->storageType == self::STORE_REMOTE) {
            throw new Exception("This storage medium has not yet been built. Please do not use");
        }
        if ($this->storageType == self::STORE_LOCAL) {
            if (!isset($this->storageKey)) {
                throw new Exception("You must define a storage key when storing locally");
            }
        }

        $view = $this->getView();
        $view->registerJsFile('https://unpkg.com/grapesjs');
        if ($this->isNewsletter) {
            $view->registerJsFile('https://unpkg.com/grapesjs-preset-newsletter');
        }
        $view->registerCssFile('https://unpkg.com/grapesjs/dist/css/grapes.min.css');

        if ($this->debug) {
            $view->registerJs($this->JS());
            $view->registerCss($this->CSS());
        } else {
            $view->registerJs($this->minify_js($this->JS()));
            $view->registerCss($this->minify_css($this->CSS()));
        }
    }


    /**
     * {@inheritdoc}
     * 
     * @return  string
     * 
     * @access  public
     */

    public function run(): string {
        $html = '';
        $html .= Html::tag('div', $this->content, options: $this->options);
        if ($this->storageType == self::STORE_LOCAL_INLINE) {
            $html .= Html::input('hidden', 'grapeJS-content-data', options: ['id' => "grapeJS-content-data{$this->id}"]);
            $html .= Html::input('hidden', 'grapeJS-content-html', options: ['id' => "grapeJS-content-html{$this->id}"]);
        }
        return $html;
    }


    /**
     * Returns the JS used for loading the editor.
     * 
     * @return  string
     * 
     * @access  public
     */

    public function JS(): string {
        if (isset($this->customLoadScript)) {
            return $this->customLoadScript;
        }
        $nl = $this->isNewsletter ? '1' : '0';

        $fromElement = <<<JS
        false
        JS;
        if ($this->fromElement) {
            $fromElement = <<<JS
            true
            JS;
        }

        $js = '';

        $storage = match ($this->storageType) {
            self::DONT_STORE => <<<JS
            false
            JS,
            /*************************************/
            self::STORE_LOCAL => <<<JS
            {
                type: 'local',
                autosave: true,
                autoload: true,
                stepsBeforeSave: 1,
                options: {
                    local: {
                    key: '{$this->storageKey}', // The key for the local storage
                    },
                }
            }
            JS,
            /*************************************/
            ## To be built
            self::STORE_REMOTE => <<<JS
            false
            JS,
            /*************************************/
            self::STORE_LOCAL_INLINE => <<<JS
            { type: 'inline' }
            JS,
                /*************************************/
            default => <<<JS
            false
            JS,
        };

        $stL = '0';
        if ($this->storageType == self::STORE_LOCAL_INLINE) {
            $stL = '1';
            $js .= <<<JS
            const inlineStorage{$this->id} = editor => {
                const projectDataEl = document.querySelector('#grapeJS-content-data{$this->id}');
                const projectHtmlEl = document.querySelector('#grapeJS-content-html{$this->id}');

                editor.Storage.add('inline', {
                    load() {
                        return JSON.parse(projectDataEl.value || '{}');
                    },
                    store(data) {
                        const component = editor.Pages.getSelected().getMainComponent();
                        projectDataEl.value = JSON.stringify(data);
                        projectHtmlEl.value = '<html>' + '<head>' + '<style>' + editor.getCss({ component }) + '</style>' + '</head>' + editor.getHtml({ component }) + '<html>';
                    }
                });
            };
            JS;
        }

        $js .= <<<JS
        const plugins{$this->id} = [];
        const pluginsOpts{$this->id} = {};
        if ({$nl} == 1) {
            plugins{$this->id}.push('grapesjs-preset-newsletter');
            pluginsOpts{$this->id}['grapesjs-preset-newsletter'] = {};
        }
        if ($stL == 1) {
            plugins{$this->id}.push(inlineStorage{$this->id});
        }

        const data{$this->id} = {
            container: '#{$this->id}',
            fromElement: {$fromElement},
            height: '{$this->height}',
            width: '{$this->width}',
            storageManager: $storage,
            panels: { defaults: [] },
            plugins: plugins{$this->id},
            pluginsOpts: pluginsOpts{$this->id},
        };

        const editor{$this->id} = grapesjs.init(data{$this->id});
        if ({$nl} == 1) {
            const pnm = editor{$this->id}.Panels;
            editor{$this->id}.on('load', function() {
                pnm.getButton('options', 'sw-visibility').set('active', 1);
            });
        }

        {$this->extraScripts}
        JS;

        return $js;
    }


    /**
     * The styles of the builder.
     * 
     * @return  string
     * 
     * @access  public
     */

    public function CSS(): string {
        if (isset($this->customStyles)) {
            return $this->customStyles;
        }
        return <<<CSS
        .grapes-js {
            border: 3px solid #444;
            border-radius: var(--panel-border-radius);
        }

        .gjs-block {
            width: auto;
            height: auto;
            min-height: auto;
        }
        CSS;
    }
}
