<?php

namespace onespace\widgets\ui\banner\base;

use app\models\Estate;
use Exception;
use onespace\widgets\ui\banner\base\BannerTemplate;
use onespace\widgets\ui\banner\base\TopBanner;
use onespace\widgets\ui\EstateLogo;
use onespace\widgets\ui\ProfilePic;
use Yii;
use yii\widgets\Pjax;

/**
 * Draw the top banner with estate links. It contains user's personal information with narrow margins.
 * 
 * @author  Gareth Palmer <gareth@one-space.co.za>
 */

abstract class EstatesBaseBanner extends TopBanner implements BannerTemplate {

    /**
     * The type of estate banner, estate management, erf owner or both.
     * 
     * @var string  $type
     * 
     * @access  protected
     */

    protected string $type;

    /**
     * The id of the PJAX element if you need to specify one. Otherwise it'll randomly generate.
     * 
     * @var string  $pjax_id
     * 
     * @access  public
     */

    public string $pjax_id;


    /**
     * {@inheritdoc}
     * 
     * @access  public
     */

    public function init(): void {
        parent::init();
        $this->pjax_id ??= 'pj' . Yii::$app->security->generateRandomString(8);
    }


    /**
     * {@inheritdoc}
     * 
     * @return  string
     * 
     * @access  public
     */

    public function run(): string {
        if (!isset($this->type)) {
            throw new Exception('$this->type not set');
        }
        Pjax::begin([
            'id' => $this->pjax_id,
            'linkSelector' => '.narrow-banner a',
        ]);
        $bar = parent::run();
        if (isset($this->container_class)) {
            $bar .= " {$this->container_class}";
        }
        $bar .= " estates-banner narrow-banner'>";

        $uri = '/' . Yii::$app->request->pathInfo;

        $sel = !Yii::$app->request->get('id') ? ' estate-selected' : '';
        $bar .= ProfilePic::widget([
            'options' => ['class' => "profile-pic-img{$sel}"],
            'link'    => ['href' => $uri, 'title' => 'All estates'],
        ]);
        $bar .= '<div class="pp-vertical-rule"></div>';

        $estates = [];
        /**
         * This is a catch is needed because this repo may move ahead of Central. If so, an empty array is used.
         */
        if (method_exists(Estate::class, 'getCurrentAdminUserEstates') && method_exists(Estate::class, 'getCurrentAdminUserEstates')) {
            $estates = match ($this->type) {
                'admin' => [Estate::getCurrentAdminUserEstates()],
                'erf'   => [Estate::getCurrentErfUserEstates()],
                'both' => [
                    Estate::getCurrentAdminUserEstates(),
                    Estate::getCurrentErfUserEstates(),
                ],
            };
        }


        $contained = [];
        foreach ($estates as $entry) {
            foreach ($entry as $estate) {
                if (in_array($estate->id, $contained)) {
                    continue;
                }
                $contained[] = $estate->id;
                $sel = Yii::$app->request->get('id') == $estate->id ? ' estate-selected' : '';
                $bar .= EstateLogo::widget([
                    'id'       => $estate->id,
                    'options'  => ['class' => "profile-pic-img{$sel}"],
                    'link'     => ['href' => $uri . "?id={$estate->id}", 'title' => $estate->name],
                    'fallback' => 'house',
                ]);
            }
        }
        $bar .= "</div>"; // container
        Pjax::end();
        return $bar;
    }


    /**
     * @inheritdoc
     */

    public function banner_css(): string {
        return $this->minify_css(<<<CSS
        .estates-banner {
            border-top-left-radius: var(--panel-border-radius);
            border-top-right-radius: var(--panel-border-radius);
            padding: 0.5rem;
            padding-left: 3.5rem;
            display: flex;
            column-gap: 1rem;
            justify-content: flex-start;
            align-items: stretch;
        }

        .narrow-banner {
            height: max-content;
        }

        .profile-pic-img {
            border: 1.5px solid var(--cool-grey);
            border-radius: 50%;
            height: 60px;
            width: 60px;
        }

        .profile-pic-img:hover {
            outline: 2px solid var(--main-accent);
        }

        .estate-selected {
            border: 2.5px solid var(--main-accent);
        }

        .pp-vertical-rule {
            width: 1px;
            border-radius: 2px;
            background-color: var(--cool-grey);
            opacity: 80%;
        }
        CSS);
    }
}
