<?php

namespace onespace\widgets\ui;

use onespace\widgets\assets\GetView;
use onespace\widgets\icons\FontAwesomeIcon;
use onespace\widgets\ui\base\Tools;
use Yii;
use yii\base\Widget;


/**
 * Generates a standard User Portal button.
 * 
 * @author  Gareth Palmer <gareth@one-space.co.za>
 */

class PortalButton extends Widget {

    use GetView;
    use Tools;


    /**
     * The icon to place onto the screen.
     * 
     * @var string  $icon   Default: cat
     * 
     * @access  public
     */

    public string $icon = 'cat';

    /**
     * The default text to place under the icon.
     * 
     * @var string  $text   Default: 'Placeholder Cat'
     * 
     * @access  public
     */

    public string $text = 'Placeholder Cat';

    /**
     * The colour of the icon and text, default is #254C5A - Corporate Blue (official Onespace branding colour)
     * 
     * @var string  $colour Default: #254C5A
     * 
     * @access  public
     */

    public string $colour = '#254C5A';

    /**
     * The styling of the icon, options are:
     * - solid
     * - regular
     * - brand
     * - light
     * 
     * @var string  $style  Default: light
     * 
     * @access  public
     */

    public string $style = 'light';

    /**
     * The size of the icon.
     * 
     * @var string  $size   Default: 5em
     * 
     * @access  public
     */

    public string $size = '5em';

    /**
     * The default link of the icon.
     * 
     * @var string  $href   Default: javascript:void(0)
     * 
     * @access  public
     */

    public string $href = 'javascript:void(0)';

    /**
     * The html title to show to the user on hover
     * 
     * @var string  $title
     * 
     * @access  public
     */
    public string $title;

    /**
     * Whether or not to place a border on top of the icon.
     * 
     * @var boolean $border_top Default: false
     * 
     * @access  public
     */

    public bool $border_top = false;

    /**
     * Whether or not to place a border on the bottom of the icon.
     * 
     * @var boolean $border_bottom  Default: false
     * 
     * @access  public
     */

    public bool $border_bottom = false;

    /**
     * Whether or not to place a border on the left of the icon.
     * 
     * @var boolean $border_left    Default: false
     * 
     * @access  public
     */

    public bool $border_left = false;

    /**
     * Whether or not to place a border on the right of the icon.
     * 
     * @var boolean $border_right   Default: false
     * 
     * @access  public
     */

    public bool $border_right = false;

    /**
     * Whether or not the icon should be disabled, but still visable to the user.
     * 
     * @var boolean $disabled   Default: false
     * 
     * @access  public
     */

    public bool $disabled = false;

    /**
     * The id of the icon. Generated as a random string.
     * 
     * @var string  $id
     */

    private string $id;

    /**
     * @inheritdoc
     */

    public function run(): string {
        parent::run();

        $this->id = Yii::$app->security->generateRandomString(12);

        $view = self::getPageView();
        $view->registerCss($this->css());

        $btn = '';

        $div_class = [
            'text-center',
            'portal-icon-container',
        ];

        if ($this->border_top) {
            $div_class[] = 'pt-bt';
        }
        if ($this->border_bottom) {
            $div_class[] = 'pt-bb';
        }
        if ($this->border_left) {
            $div_class[] = 'pt-bl';
        }
        if ($this->border_right) {
            $div_class[] = 'pt-br';
        }
        if ($this->disabled) {
            $div_class[] = 'disabled';
            $this->href = 'javascript:void(0)';
            $this->colour = '#aaa';
            $this->title ??= 'Coming soon';
        }

        if (!isset($this->title)) {
            $this->title = $this->text;
        }

        $div_class = implode(' ', $div_class);

        $btn .= "<div class='{$div_class}' id='{$this->id}'>";
        $btn .= "<a href='{$this->href}' class='portal-icon-link' style='color:{$this->colour}'";
        if (isset($this->title)) {
            $btn .= " title='{$this->title}'";
        }
        $btn .= ">";

        $btn .= FontAwesomeIcon::widget([
            'identifier' => $this->icon,
            'style' => $this->style,
            'colour' => $this->colour,
            'content' => '',
            'options' => [
                'style' => "font-size:{$this->size};",
                'id' => $this->id . '-icon',
                'class' => 'portal-icon',
            ],
        ]);

        $btn .= "<div class='portal-text'>";
        $btn .= $this->text;
        $btn .= "</div>";

        $btn .= '</a>';
        $btn .= '</div>';

        return $btn;
    }


    /**
     * The CSS used in styling the icon.
     * 
     * @return  string
     * 
     * @access  private
     */

    private function css(): string {
        return $this->minify_css(<<<CSS
        :root {
            --sub-border: 1px solid #cfd2d1
        }

        .portal-icon-container {
            padding: 3px;
        }

        .portal-icon-container:hover:not(.disabled) {
            transform: scale(1.08);
        }

        .disabled,
        .disabled a {
            cursor: no-drop;
        }

        .portal-icon-link {
            text-decoration: none;
        }

        .portal-text {
             margin-top: 2.5rem;
            font-size: 2rem;
        }

        .portal-icon {
            text-decoration:none;
            display:block; 
        }

        .pt-bl {
            border-left: var(--sub-border);
        }
        .pt-br {
            border-right: var(--sub-border);
        }
        .pt-bt {
            border-top: var(--sub-border);
        }
        .pt-bb {
            border-bottom: var(--sub-border);
        }
        CSS);
    }

}
