<?php

namespace onespace\widgets\dropzone;

use Exception;
use Yii;
use yii\base\Widget;
use yii\helpers\FileHelper;
use yii\web\UploadedFile;
use yii\web\View;

/**
 * Insert a Dropzone file uploader as required.
 * 
 * Widget wrapper for Dropzone.
 * 
 * @see https://docs.dropzone.dev/
 * 
 * @author Gareth Palmer <gareth@one-space.co.za>
 */

final class Dropzone extends Widget {

    /**
     * REQUIRED. The ID of the uploader form
     *
     * @var string  $id
     * 
     * @access  public
     */

    public string $id;

    /**
     * REQUIRED. The URI that will handle 
     *
     * @var string  $action
     * 
     * @access  public
     */
    public string $action;

    public bool $chunking =  true;
    public int $chunkSize =  2000000;
    public bool $forceChunking =  false;
    public bool $retryChunks =  false;
    public int $retryChunksLimit =  3;
    public int $parallelUploads =  3;
    public bool $parallelChunkUploads =  false;
    public int $timeout =  120000;
    public int $maxFilesize =  5000000000;

    /**
     * OPTIONAL. Set a specific type of file that can be uploaded. Leave null to remove this limit.
     * 
     * Must be parsed as a MIME-TYPE or as a extension (eg .docx).
     * 
     * Multiple values must be seperated with a comma `,`
     * 
     * @note    There is an interesting bug, driving from the browser in the Windows context:
     *          If you wish to limit to a CSV file, the MIME-TYPE is `text/csv`. If this property
     *          is set to `text/csv` and you have MS Office installed, it'll be detected as 
     *          `application/vnd.ms-excel`. Instead pass `.csv` to limit to this MIME-TYPE.
     *          See: https://christianwood.net/posts/csv-file-upload-validation/ for more info.
     * 
     * @see https://developer.mozilla.org/en-US/docs/Web/HTTP/Basics_of_HTTP/MIME_types/Common_types
     * @see https://docs.dropzone.dev/configuration/basics/configuration-options#acceptedFiles
     *
     * @var string|null $acceptedFiles  Default: null
     * 
     * @access  public
     */
    public ?string $acceptedFiles = null;

    /**
     * OPTIONAL. Any other params to add to the upload form element.
     *
     * @var array   $options
     * 
     * @access  public
     */
    public array $options = [];

    /**
     * OPTIONAL. Any hidden inputs to add to the upload form.
     *
     * @var array   $hiddenInputs
     * 
     * @access  public
     */
    public array $hiddenInputs = [];


    /**
     * {@inheritDoc}
     */

    public function init() {
        parent::init();

        if (!isset($this->id)) {
            throw new Exception("You must specify an element id from which the dropzone must hook.");
        }
        if (!isset($this->action)) {
            throw new Exception("You must specify an action, the URI which will handle any files uploaded.");
        }

        if (isset($this->options['id'])) {
            $this->id = $this->options['id'];
            unset($this->options['id']);
        }
        if (isset($this->options['class'])) {
            $this->options['class'] = 'dropzone ' . $this->options['class'];
        } else {
            $this->options['class'] = 'dropzone';
        }

        $this->options['enctype'] = 'multipart/form-data';
        $this->options['method'] = 'POST';
        $this->options['action'] = $this->action;

        $this->hiddenInputs['_csrf'] ??= Yii::$app->request->csrfToken;

        $view = $this->getView();
        $view->registerCss(file_get_contents(Yii::getAlias('@vendor/enyo/dropzone/dist/min/dropzone.min.css')));
        $view->registerJs(file_get_contents(Yii::getAlias('@vendor/enyo/dropzone/dist/min/dropzone.min.js')), View::POS_HEAD);

        $accept = is_null($this->acceptedFiles) ? 'NULL' : $this->acceptedFiles;
        $rand_id = str_replace('-', '_', Yii::$app->security->generateRandomString(8));

        $view->registerJs(<<<JS
        const chunking{$rand_id} = '$this->chunking' == '1' ? true : false;
        const forceChunking{$rand_id} = '$this->forceChunking' == '1' ? true : false;
        const retryChunks{$rand_id} = '$this->retryChunks' == '1' ? true : false;
        const parallelChunkUploads{$rand_id} = '$this->parallelChunkUploads' == '1' ? true : false;
        const accept{$rand_id} = '$accept' == 'NULL' ? null : '$accept';
        Dropzone.autoDiscover = false;
        const dropzone{$rand_id} = new Dropzone("form#{$this->id}", {
            url: '{$this->action}',
            chunking: chunking{$rand_id},
            chunkSize: $this->chunkSize,
            forceChunking: forceChunking{$rand_id},
            retryChunks: retryChunks{$rand_id},
            retryChunksLimit: $this->retryChunksLimit,
            parallelUploads: $this->parallelUploads,
            parallelChunkUploads: parallelChunkUploads{$rand_id},
            timeout: $this->timeout,
            maxFilesize: $this->maxFilesize,
            acceptedFiles: accept{$rand_id},
            init: function () {
                this.on("sending", function (file, xhr, formData) {
                    // xhr.onreadystatechange = function () {
                    //     if (this.readyState == 4) {
                    //         console.log('While Sending:', JSON.parse(this.responseText));
                    //     }
                    // };
                    xhr.ontimeout = (function () {
                        alert('Error: Server Timeout');
                    });
                }).on("success", function (res) {
                    let response = JSON.parse(res.xhr.response);
                    // console.log('On Success', response);
                    if (response.status == "error") {
                        alert(response.info);
                    }
                }).on("error", function (file, response) {
                    if (response.message !== undefined) {
                        response = reponse.message
                    }
                    alert("Error: " + response);
                });
            }
        });
        JS, View::POS_END);
    }


    /**
     * {@inheritDoc}
     * 
     * @return string
     * 
     * @access  public
     */

    public function run(): string {
        $html = '';
        $html .= parent::run();

        $html .= "<form id='{$this->id}'";
        foreach ($this->options as $key => $value) {
            $html .= " {$key}='{$value}'";
        }
        $html .= ">";
        foreach ($this->hiddenInputs as $name => $value) {
            $html .= "<input type='hidden' name='{$name}' value='{$value}'>";
        }
        $html .= "</form>";
        return $html;
    }


    /**
     * Method for handling uploads. It should be called in your upload handling controller action.
     * 
     * It automatically handles chunked uploads if that is required.
     * 
     * @param   string  $upload_dir
     * 
     * @return string
     * 
     * @access  public
     */

    public static function handle_upload(string $upload_dir): string {
        /**
         * Ensure the upload dir is the correctly formatted.
         * 
         * @example /var/www/html/upload/directory -> /var/www/html/upload/directory/
         */
        $upload_dir = rtrim($upload_dir, '/') . '/';

        /**
         * Create the upload directory if it does not exist.
         */
        if (!file_exists($upload_dir)) {
            mkdir($upload_dir, 0777, true);
        }

        /**
         * Perform the initial upload to /tmp
         */
        $upload = UploadedFile::getInstanceByName('file');

        /**
         * Determine if the upload is being chunked.
         */
        $request = Yii::$app->request;
        $chunkNumber = $request->post('dzchunkindex');

        if ($request->post('dzuuid') !== null) {
            /**
             * CHUNKED
             * 
             * Start by setting a temporary chunk file.
             */
            $chunksPath = $upload_dir;
            $tempFilename = $request->post('dzuuid');
            $tempFilePath = $chunksPath . $tempFilename . '_' . $chunkNumber;

            /**
             * Save the chunk temporarily
             */
            $upload->saveAs($tempFilePath);

            /**
             * Determine if this chunk is the last run
             */
            if ((int)$request->post('dzchunkindex') === (int)($request->post('dztotalchunkcount') - 1)) {
                /**
                 * Process and merge the files and write it to the desired final file.
                 */
                $file_name = $upload_dir . $upload->name;
                $chunkFiles = FileHelper::findFiles($chunksPath);
                natcasesort($chunkFiles);
                $finalFile = fopen($file_name, 'ab');
                foreach ($chunkFiles as $file) {
                    if (!str_contains($file, $tempFilename)) {
                        continue;
                    }
                    $chunkData = file_get_contents($file);
                    fwrite($finalFile, $chunkData);
                    unlink($file); // Delete the chunk file
                }
                fclose($finalFile);
                $response = [
                    'code' => 200,
                    'status' => 'Upload Complete',
                    'path' => $file_name,
                ];
            } else {
                $response = [
                    'code' => 206,
                    'status' => 'Partial chunk upload complete',
                    'chunk_file' => $tempFilePath,
                ];
            }
            return json_encode($response);
        } else {
            /**
             * NOT CHUNKED
             * 
             * Save the file and place it in the desired directory.
             */
            $fileLocation = $upload->baseName . '.' . $upload->extension;
            if (!$upload->saveAs($upload_dir . $fileLocation, false)) {
                $response = ['code' => 500];
            } else {
                $response = [
                    'code' => 200,
                    'status' => 'Upload Complete',
                    'path' => $fileLocation,
                ];
            }
            return json_encode($response);
        }
    }
}
