<?php

namespace onespace\widgets\ui\views\datacard;

use onespace\widgets\ui\base\Tools;
use yii\base\Widget;

/**
 * Widget for the datacard details interface.
 * 
 * @author  Gareth Palmer <gareth@one-space.co.za>
 */

final class ExpandedDetails extends Widget {

    use Tools;

    /**
     * The image to put on the page.
     * 
     * @var string|null $image
     * 
     * @access  public
     */

    public ?string $image;

    /**
     * The fields to place into the data columns under the image.
     * 
     * Should be formatted as follows:
     * 
     * ```php
     * [
     *  'Table Key' => 'Table Value',
     *  ...
     * ];
     * ```
     * 
     * Both `Table Key` & `Table Value` will be printed on the page.
     * 
     * @var array   $fields
     * 
     * @access  public
     */

    public array $fields;


    /**
     * {@inheritdoc}
     */

    public function init() {
        parent::init();

        $this->fields ??= [];

        $view = $this->getView();
        $view->registerCss($this->styles());
    }


    /**
     * {@inheritdoc}
     * 
     * @return  string
     * 
     * @access  public
     */

    public function run(): string {
        parent::run();
        $card = '';

        $card .= "<div class='datacard-data-container'>";
        $card .= $this->image();
        $card .= "<div class='datacard-info-container'>";
        foreach ($this->fields as $heading => $value) {
            $card .= $this->data_column($heading, $value ?? '');
        }
        $card .= "</div>"; // datacard-info-container
        $card .= "</div>"; // datacard-data-container

        return $card;
    }


    /**
     * Return the image portion of the data cart details page.
     * 
     * @return  string
     * 
     * @access  public
     */

    public function image(): string {
        $img = '';
        $image_path = $this->image ?? '';
        if (substr($image_path, 0, 2) == '<i') {
            // PLACEHOLDER IMAGE
            $img .= "<div class='datacard-image-container-placeholder'>";
            $img .= $image_path;
        } else {
            // USER UPLOADED IMAGE
            $img .= "<div class='datacard-image-container'>";
            $img .= "<img src='{$image_path}' class='datacard-image'>";
        }
        $img .= "</div>"; // datacard-image-container-placeholder / datacard-image-container
        return $img;
    }


    /**
     * Draw a data column block
     * 
     * @param   string  $heading
     * @param   string  $value
     * 
     * @return  string
     * 
     * @access  public
     */

    private function data_column(string $heading, string $value): string {
        $col = '';
        $col .= "<div class='datacard-info-details'>";
        $col .= "<div class='datacard-info-details-heading'>";
        $col .= $heading;
        $col .= "</div>"; // datacard-info-details-heading
        $col .= "<div class='datacard-info-details-value'>";
        $col .= $value;
        $col .= "</div>"; // datacard-info-details-value
        $col .= "</div>"; // datacard-info-details
        return $col;
    }


    /**
     * Generate the required styles for the explained view.
     * 
     * @return  string
     * 
     * @access  private
     */

    private function styles(): string {
        return $this->minify_css(<<<CSS
        .datacard-padding-overwrite {
            padding: unset;
            /* border: 2px solid var(--main-accent); */
        }

        .datacard-data-container {
            display: grid;
            grid-template-columns: 1fr;
            grid-template-rows: 350px 1fr;
        }

        .datacard-image-container-placeholder {
            place-self: center;
        }

        .datacard-image-container {
            overflow: hidden;
        }

        .datacard-image {
            width: 100%;
            height: 100%;
            object-fit: cover;
        }

        .datacard-info-container {
            padding: var(--panel-padding);
            display: grid;
            grid-template-columns: repeat(2, 1fr);
            grid-template-rows: auto;
            grid-gap: 1rem;
            /* border: 2px solid var(--main-accent); */
        }

        .datacard-info-details {
            display: grid;
            grid-template-columns: 2.5fr 3fr;
            grid-template-rows: 1fr;
            font-size: 1.5rem;
        }

        .datacard-info-details-heading {
            margin: 1rem 0;
            font-weight: 600;
            color: var(--smokey-grey);
        }

        .datacard-info-details-value {
            margin: 1rem 0;
            font-weight: 600;
            color: #000;
        }
        CSS);
    }
}
