<?php

namespace onespace\widgets\ui\views\datacard;

use Exception;
use onespace\widgets\icons\FontAwesomeIcon;
use yii\base\Widget;
use yii\web\View;

/**
 * Widget for drawing an interface card link.
 * 
 * @author  Gareth Palmer <gareth@one-space.co.za>
 */

final class DataCard extends Widget {

    /**
     * REQUIRED.
     * 
     * The link the card should load.
     * 
     * @var string  $link
     * 
     * @access  public
     */

    public string $link;

    /**
     * OPTIONAL.
     * 
     * The image to draw onto the card.
     * 
     * @var string|null $image
     * 
     * @access  public
     */

    public ?string $image;

    /**
     * OPTIONAL.
     * 
     * The heading of the card.
     * 
     * @var string  $heading
     * 
     * @access  public
     */

    public string $heading;

    /**
     * OPTIONAL.
     * 
     * The text details of the card.
     * 
     * @var string  $details
     * 
     * @access  public
     */

    public string $details;

    /**
     * OPTIONAL.
     * 
     * The `edit` link. Set to false to switch off.
     * 
     * @var string|false    $edit
     * 
     * @access  public
     */

    public string|false $edit = false;

    /**
     * OPTIONAL.
     * 
     * The `delete` link. Set to false to switch off.
     * 
     * @var string|false    $delete
     * 
     * @access  public
     */

    public string|false $delete = false;

    /**
     * Whether the card is in a selected state.
     * 
     * @var bool    $selected   Default: false
     * 
     * @access  public
     */

    public bool $selected = false;


    /**
     * {@inheritdoc}
     * 
     * @throws  Exception
     */

    public function init() {
        parent::init();
        if (!isset($this->link)) {
            throw new Exception("A link must be set on a DataCard");
        }

        $view = $this->getView();
        $view->registerJs(<<<JS
            function naved(event, loc, confirm = false) {
                event.preventDefault();
                if (confirm && !window.confirm("Are you sure?")) {
                    return;
                }
                window.location.href = loc;
            }
        JS, View::POS_END);
    }


    /**
     * {@inheritdoc}
     * 
     * @return  string
     */

    public function run(): string {
        parent::run();

        $card = '';

        $sel = $this->selected ? " data-selected='1'" : '';
        $card .= "<div class='ui-card-container'{$sel}>";

        $card .= "<a href='{$this->link}'>";
        $card .= "<div class='ui-card-elements'>";

        $card .= $this->image();
        $card .= $this->heading();
        $card .= $this->details();
        $card .= "<div class='ui-card-buttons'>";
        $card .= $this->edit();
        $card .= $this->delete();
        $card .= "</div>"; // ui-card-buttons

        $card .= '</div>'; // ui-card-elements
        $card .= '</a>';

        $card .= '</div>'; // ui-card-container
        return $card;
    }


    /**
     * Draws the image on the card.
     * 
     * @return  string
     * 
     * @access  private
     */

    private function image(): string {
        $image = '';
        $image .= "<div class='ui-card-image'>";

        if (isset($this->image)) {
            if (substr($this->image, 0, 2) == '<i') {
                $image .= $this->image;
            } else {
                $image .= "<img src='{$this->image}' height='140px'>";
            }
        }

        $image .= '</div>'; // ui-card-image
        return $image;
    }


    /**
     * Draw the card heading.
     * 
     * @return  string
     * 
     * @access  private
     */

    private function heading(): string {
        $heading = '';
        $heading .= "<div class='ui-card-heading'>";

        if (isset($this->heading)) {
            $heading .= "<b>{$this->heading}</b>";
        }

        $heading .= '</div>'; // ui-card-heading
        return $heading;
    }


    /**
     * Draw the card details text.
     * 
     * @return  string
     * 
     * @access  private
     */

    private function details(): string {
        $details = '';
        $details .= "<div class='ui-card-details'>";

        if (isset($this->details)) {
            $details .= $this->details;
        }

        $details .= '</div>'; // ui-card-details
        return $details;
    }


    /**
     * Draw the edit link onto the card.
     * 
     * @return  string
     * 
     * @access  private
     */

    private function edit(): string {
        $edit = '';
        $edit .= "<div class='ui-card-edit ui-data-card-context-button' onclick='naved(event, \"{$this->edit}\")'>";

        if ($this->edit !== false) {
            $edit .= FontAwesomeIcon::widget([
                'identifier' => 'edit',
                'style' => 'regular',
                'options' => ['style' => 'color: var(--main-accent)'],
            ]);
        }

        $edit .= '</div>'; // ui-card-edit
        return $edit;
    }


    /**
     * Draw the delete link onto the card.
     * 
     * @return  string
     * 
     * @access  private
     */

    private function delete(): string {
        $delete = '';
        $delete .= "<div class='ui-card-delete ui-data-card-context-button' onclick='naved(event, \"{$this->delete}\", true)'>";

        if ($this->delete !== false) {
            $delete .= FontAwesomeIcon::widget([
                'identifier' => 'trash-alt',
                'style' => 'regular',
                'options' => ['style' => 'color: var(--main-accent)'],
            ]);
        }

        $delete .= '</div>'; // ui-card-delete
        return $delete;
    }
}
