<?php

namespace onespace\widgets\ui;


use onespace\widgets\icons\FontAwesomeIcon;
use onespace\widgets\ui\base\Align;
use yii\base\Widget;

/**
 * Draw out the standard OneSpaceGridview header, with a number of configurable options.
 * 
 * @see src/ui/OneSpaceGridview.php
 * 
 * @author  Gareth Palmer <gareth@one-space.co.za>
 */
class ItemHeader extends Widget {

    /**
     * If desired, the title to be placed into the header. This is just a plain text / html header.
     * 
     * @var string  $title
     * 
     * @access  public
     */

    public string $title;

    /**
     * The desired alignment of the title on the header.
     * 
     * @var Align   $title_position
     * 
     * @access  public
     */

    public Align $title_position = Align::LEFT;

    /**
     * If desired, the button to be placed into the header. This value is specifically the text of the button.
     * 
     * @var array   $button
     * 
     * @access  public
     */

    public array $button;

    /**
     * The desired alignment of the button on the header.
     * 
     * @var Align   $button_position
     * 
     * @access  public
     */

    public Align $button_position = Align::CENTER;

    /**
     * If desired, the filter to be placed into the header. This value is specifically the id of the filter row, which should be hidden / unhidden.
     * 
     * @var string  $filter
     * 
     * @access  public
     */

    public string $filter;

    /**
     * The desired alignment of the filter on the header.
     * 
     * @var Align   $filter_position
     * 
     * @access  public
     */

    public Align $filter_position = Align::RIGHT;

    /**
     * If the alignment does not sufficently lay out the elements as desired, you can parse desired CSS here.
     * 
     * Specifically the value to be placed as a `grid-template-columns` property.
     * 
     * ## Examples
     * 
     * ```php
     *  ... 
     *  'grid_template_columns' => 'repeat(3, 1fr)', // 3 equal elements.
     *  'grid_template_columns' => 'max-content 1fr', // 1 Elements that is max content size, then 1 which fits the rest of the space.
     *  'grid_template_columns' => '200px 300px 200px', // An element of 200px, then one of 300px, then one of 200px again.
     *  ...
     * ```
     * 
     * @var string  $grid_template_columns
     * 
     * @access  public
     */

    public string $grid_template_columns;

    /**
     * Added param for drawing a `<hr>` on the bottom of the filter.
     * 
     * @var bool    $hr Default: false
     * 
     * @access  public
     */

    public bool $hr = false;


    /**
     * @inheritdoc
     * 
     * @return  string
     */

    public function run(): string {
        parent::run();
        $bar = '';
        $bar .= "<div class='item-header-container'";
        if (isset($this->grid_template_columns)) {
            $bar .= " style='grid-template-columns: {$this->grid_template_columns}'";
        }
        $bar .= ">";
        $bar .= $this->set_title();
        $bar .= $this->set_button();
        $bar .= $this->set_filter();
        $bar .= '</div>';
        if ($this->hr) {
            $bar .= "<hr class='ui-item-hr'>";
        }
        return $bar;
    }


    /**
     * If the title is set, this places it on the page.
     * 
     * @return  string
     * 
     * @access  private
     */

    private function set_title(): string {
        if (!isset($this->title)) {
            return '';
        }
        $item = "<div class='ui-item-header-text {$this->title_position->get_class()}'>";
        $item .= $this->title;
        $item .= "</div>";
        return $item;
    }


    /**
     * If the button is set, this places it on the page.
     * 
     * @return  string
     * 
     * @access  private
     */

    private function set_button(): string {
        if (!isset($this->button)) {
            return '';
        }
        $item = "<div class='{$this->button_position->get_class()}'>";

        if (isset( $this->button['class'])) {
            $this->button['class'] = 'btn ui-btn-action ' . $this->button['class'];
        } else { 
            $this->button['class'] = 'btn ui-btn-action';
        }
        $this->button['href'] ??= 'javascript:void(0)';
        $this->button['onclick'] = "window.location.href = \"{$this->button['href']}\"";
        unset($this->button['href']);

        $item .= '<button';
        foreach ($this->button as $key => $value) {
            $item .= " {$key}='{$value}'";
        }
        $item .= ">{$this->button['value']}</button>";

        $item .= "</div>";
        return $item;
    }


    /**
     * If the filter is set, this places it on the page.
     * 
     * @return  string
     * 
     * @access  private
     */

    private function set_filter(): string {
        if (!isset($this->filter)) {
            return '';
        }
        $item = "<div class='{$this->filter_position->get_class()}'>";
        $options = ['class' => 'ui-filter-button'];
        if (isset($this->filter)) {
            $options['onclick'] = "document.getElementById('{$this->filter}').classList.toggle('hidden')";
        }
        $item .= FontAwesomeIcon::widget([
            'identifier' => 'filter',
            'style'      => 'regular',
            'options'    => $options,
        ]);
        $item .= "</div>";
        return $item;
    }
}
