<?php

/**
 * @link https://www.yiiframework.com/
 * @copyright Copyright (c) 2008 Yii Software LLC
 * @license https://www.yiiframework.com/license/
 */

namespace onespace\tools\components\azuredb;

use MicrosoftAzure\Storage\Table\Models\QueryEntitiesOptions;
use yii\base\InvalidCallException;
use yii\helpers\ArrayHelper;

/**
 * DataReader represents a forward-only stream of rows from a query result set.
 *
 * To read the current row of data, call [[read()]]. The method [[readAll()]]
 * returns all the rows in a single array. Rows of data can also be read by
 * iterating through the reader. For example,
 *
 * ```php
 * $command = $connection->createCommand('SELECT * FROM post');
 * $reader = $command->query();
 *
 * while ($row = $reader->read()) {
 *     $rows[] = $row;
 * }
 *
 * // equivalent to:
 * foreach ($reader as $row) {
 *     $rows[] = $row;
 * }
 *
 * // equivalent to:
 * $rows = $reader->readAll();
 * ```
 *
 * Note that since DataReader is a forward-only stream, you can only traverse it once.
 * Doing it the second time will throw an exception.
 *
 * It is possible to use a specific mode of data fetching by setting
 * [[fetchMode]]. See the [PHP manual](https://www.php.net/manual/en/function.PDOStatement-setFetchMode.php)
 * for more details about possible fetch mode.
 *
 * @property-read int $columnCount The number of columns in the result set.
 * @property-write int $fetchMode Fetch mode.
 * @property-read bool $isClosed Whether the reader is closed or not.
 * @property-read int $rowCount Number of rows contained in the result.
 *
 * @author Qiang Xue <qiang.xue@gmail.com>
 * @since 2.0
 */
class DataReader extends \yii\base\BaseObject implements \Iterator, \Countable {
    /**
     * @var QueryEntitiesOptions the QueryEntitiesOptions associated with the command
     */
    private $_query;
    /**
     * @var Connection
     */
    private $_adb;
    private $_table;
    private $_closed = false;
    private $_row;
    private $_index = -1;
    private $_end = false;

    private array $_batchRows = [];


    /**
     * Constructor.
     * @param Command $command the command generating the query result
     * @param array $config name-value pairs that will be used to initialize the object properties
     */
    public function __construct(Command $command, $config = []) {
        $this->_query = $command->queryOptions;
        $this->_adb = $command->adb;
        $this->_table = $command->table;
        parent::__construct($config);
    }

    /**
     * Advances the reader to the next row in a result set.
     * @return array the current row, false if no more row available
     */
    public function read() {
        // load from the batch currently loaded if possible
        if ($this->_batchRows != null) {
            // if next returns false, we're at the end of the current batch
            if (next($this->_batchRows) != false) {
                return current($this->_batchRows);
            }
        }

        // end if we're at the end of things
        if ($this->_end) {
            return false;
        }

        // there is no batch, or we're at the end of a batch
        $rows = $this->_adb->client->queryEntities($this->_table, $this->_query);

        if ($rows->getNextPartitionKey() === null && $rows->getNextRowKey() === null) {
            $this->_end = true;
        } else {
            $this->_end = false;
            $this->_query->setNextPartitionKey($rows->getNextPartitionKey());
            $this->_query->setNextRowKey($rows->getNextRowKey());
        }
        $this->_batchRows = [];
        // convert to an easier to work with array (direct property=>value mapping)
        /** @var Entity[] $entities */
        foreach ($rows->getEntities() as $entity) {
            $responseEntity = [];
            foreach ($entity->getProperties() as $key => $property) {
                /** @var Property $property */
                $responseEntity[$key] = $property->getValue();
            }
            $this->_batchRows[] = $responseEntity;
        }
        return current($this->_batchRows);
    }

    /**
     * Returns a single column from the next row of a result set.
     * @param int $columnIndex zero-based column index
     * @return mixed the column of the current row, false if no more rows available
     */
    public function readColumn($columnIndex) {
        $rows = $this->read();
        if ($rows != false) {
            return $rows[$columnIndex];
        } else {
            return false;
        }
    }

    /**
     * Returns an object populated with the next row of data.
     * @param string $className class name of the object to be created and populated
     * @param array $fields Elements of this array are passed to the constructor
     * @return mixed the populated object, false if no more row of data available
     */
    public function readObject($className, $fields) {
        return $this->_statement->fetchObject($className, $fields);
    }

    /**
     * Reads the whole result set into an array.
     * @return array the result set (each array element represents a row of data).
     * An empty array will be returned if the result contains no row.
     */
    public function readAll() {
        $done = false;
        $entities = [];
        while (!$done) {
            $result = $this->_adb->client->queryEntities($this->_table, $this->_query);

            // check if there are more results
            if ($result->getNextPartitionKey() != null || $result->getNextRowKey() != null) {
                // get this list of results
                $entities = ArrayHelper::merge($entities, $result->getEntities());
                $this->_query->setNextPartitionKey($result->getNextPartitionKey());
                $this->_query->setNextRowKey($result->getNextRowKey());
            } else {
                if ($entities == null) {
                    $entities = $result->getEntities();
                } else {
                    $entities = ArrayHelper::merge($entities, $result->getEntities());
                }
                $done = true;
                $this->_end = true;
            }
        }
        /** @var Entity[] $entities */
        foreach ($entities as &$entity) {
            $responseEntity = [];
            foreach ($entity->getProperties() as $key => $property) {
                /** @var Property $property */
                $responseEntity[$key] = $property->getValue();
            }
            $entity = $responseEntity;
        }
        return $entities;
    }

    /**
     * Advances the reader to the next result when reading the results of a batch of statements.
     * This method is only useful when there are multiple result sets
     * returned by the query. Not all DBMS support this feature.
     * @return bool Returns true on success or false on failure.
     */
    public function nextResult() {
        return false;
    }

    /**
     * Closes the reader.
     * This frees up the resources allocated for executing this SQL statement.
     * Read attempts after this method call are unpredictable.
     */
    public function close() {
        $this->_adb->close();
        $this->_closed = true;
    }

    /**
     * whether the reader is closed or not.
     * @return bool whether the reader is closed or not.
     */
    public function getIsClosed() {
        return $this->_closed;
    }

    /**
     * Returns the number of rows in the result set.
     * Note, most DBMS may not give a meaningful count.
     * In this case, use "SELECT COUNT(*) FROM tableName" to obtain the number of rows.
     * @return int number of rows contained in the result.
     */
    public function getRowCount() {
        return $this->_statement->rowCount();
    }

    /**
     * Returns the number of rows in the result set.
     * This method is required by the Countable interface.
     * Note, most DBMS may not give a meaningful count.
     * In this case, use "SELECT COUNT(*) FROM tableName" to obtain the number of rows.
     * @return int number of rows contained in the result.
     */
    #[\ReturnTypeWillChange]
    public function count() {
        return $this->getRowCount();
    }

    /**
     * Returns the number of columns in the result set.
     * Note, even there's no row in the reader, this still gives correct column number.
     * @return int the number of columns in the result set.
     */
    public function getColumnCount() {
        return $this->_statement->columnCount();
    }

    /**
     * Resets the iterator to the initial state.
     * This method is required by the interface [[\Iterator]].
     * @throws InvalidCallException if this method is invoked twice
     */
    #[\ReturnTypeWillChange]
    public function rewind() {
        throw new InvalidCallException('DataReader cannot rewind. It is a forward-only reader.');
    }

    /**
     * Returns the index of the current row.
     * This method is required by the interface [[\Iterator]].
     * @return int the index of the current row.
     */
    #[\ReturnTypeWillChange]
    public function key() {
        return $this->_index;
    }

    /**
     * Returns the current row.
     * This method is required by the interface [[\Iterator]].
     * @return mixed the current row.
     */
    #[\ReturnTypeWillChange]
    public function current() {
        return $this->_row;
    }

    /**
     * Moves the internal pointer to the next row.
     * This method is required by the interface [[\Iterator]].
     */
    #[\ReturnTypeWillChange]
    public function next() {
        $this->_row = $this->_statement->fetch();
        $this->_index++;
    }

    /**
     * Returns whether there is a row of data at current position.
     * This method is required by the interface [[\Iterator]].
     * @return bool whether there is a row of data at current position.
     */
    #[\ReturnTypeWillChange]
    public function valid() {
        return $this->_row !== false;
    }
}
