<?php

namespace onespace\tools\components\tdengine;

use DateTime;
use Exception;
use InvalidArgumentException;
use onespace\tools\components\tdengine\ActiveQuery;
use PDO;
use Yii;
use yii\base\InvalidConfigException;
use yii\base\InvalidParamException;
use yii\base\ModelEvent;
use yii\db\ActiveQueryInterface;
use yii\db\AfterSaveEvent;
use yii\db\BaseActiveRecord;
use yii\db\StaleObjectException;
use yii\helpers\ArrayHelper;
use yii\helpers\StringHelper;

abstract class TDEngineActiveRecord extends BaseActiveRecord {

    public const PARAM_TAOS_NULL       = PDO::PARAM_TAOS_NULL;
    public const PARAM_TAOS_BOOL       = PDO::PARAM_TAOS_BOOL;
    public const PARAM_TAOS_TINYINT    = PDO::PARAM_TAOS_TINYINT;
    public const PARAM_TAOS_SMALLINT   = PDO::PARAM_TAOS_SMALLINT;
    public const PARAM_TAOS_INT        = PDO::PARAM_TAOS_INT;
    public const PARAM_TAOS_BIGINT     = PDO::PARAM_TAOS_BIGINT;
    public const PARAM_TAOS_FLOAT      = PDO::PARAM_TAOS_FLOAT;
    public const PARAM_TAOS_DOUBLE     = PDO::PARAM_TAOS_DOUBLE;
    public const PARAM_TAOS_VARCHAR    = PDO::PARAM_TAOS_VARCHAR;
    public const PARAM_TAOS_TIMESTAMP  = PDO::PARAM_TAOS_TIMESTAMP;
    public const PARAM_TAOS_NCHAR      = PDO::PARAM_TAOS_NCHAR;
    public const PARAM_TAOS_UTINYINT   = PDO::PARAM_TAOS_UTINYINT;
    public const PARAM_TAOS_USMALLINT  = PDO::PARAM_TAOS_USMALLINT;
    public const PARAM_TAOS_UINT       = PDO::PARAM_TAOS_UINT;
    public const PARAM_TAOS_UBIGINT    = PDO::PARAM_TAOS_UBIGINT;
    public const PARAM_TAOS_JSON       = PDO::PARAM_TAOS_JSON;
    public const PARAM_TAOS_VARBINARY  = PDO::PARAM_TAOS_VARBINARY;
    public const PARAM_TAOS_DECIMAL    = PDO::PARAM_TAOS_DECIMAL;
    public const PARAM_TAOS_BLOB       = PDO::PARAM_TAOS_BLOB;
    public const PARAM_TAOS_MEDIUMBLOB = PDO::PARAM_TAOS_MEDIUMBLOB;
    public const PARAM_TAOS_BINARY     = PDO::PARAM_TAOS_BINARY;
    // public const PARAM_TAOS_GEOMETRY   = ??;
    public const PARAM_TAOS_MAX        = PDO::PARAM_TAOS_MAX;


    private const PRIMARY_TIMESTAMP_NAME = 'ts';


    /**
     * Declares the name of the database table associated with this AR class.
     * By default this method returns the class name as the table name. You may override this method
     * if the table is not named after this convention.
     * @return string the table name
     */
    public static function tableName() {
        return StringHelper::basename(get_called_class());
    }

    /**
     * Define the list of columns that are stored in the DB. These definitions are used to generate base
     * validation rules, for data type conversions, etc.
     * 
     * Therefore, it is important to get right
     * 
     * @return array array of defined columns in format [name => data type]
     */
    public static abstract function columns();


    /**
     * Override the name of the PartitionKey attribute. Both names (whatever is returned by this method, and
     * the default) can be used interchangeably to refer to the attribute
     * 
     * @return string friendly attribute name
     */
    public static function getPrimaryTSName() {
        return self::PRIMARY_TIMESTAMP_NAME;
    }


    /**
     * @var array attribute values indexed by attribute names
     */
    private $_attributes = [];
    /**
     * @var array|null old attribute values indexed by attribute names.
     * This is `null` if the record [[isNewRecord|is new]].
     */
    private $_oldAttributes;
    /**
     * @var array related models indexed by the relation names
     */
    private $_related = [];
    /**
     * @var array relation names indexed by their link attributes
     */
    private $_relationsDependencies = [];



    /**
     * Returns the database connection used by this AR class.
     * By default, the "tdengineDb" application component is used as the database connection.
     * You may override this method if you want to use a different database connection.
     * @return Connection the database connection used by this AR class.
     */
    public static function getDb() {
        return Yii::$app->tdengineDb;
    }


    /**
     * Return the raw PDO object for using TAOS TDEngine.
     * 
     * Needed because some standard PDO tools used by TDEngine are
     * bugged and therefore an overwrite is needed.
     * 
     * @return PDO
     * 
     * @static
     * @access  public
     */

    public static function getPDO(): PDO {
        $taos = require Yii::getAlias('@osWidgets/config/tdengineConnection.php');
        return new PDO($taos['dsn'], $taos['username'], $taos['password']);
    }


    /**
     * Returns a Unix Timestamp to the thousandsth of a second, for inserting into TDEngine,
     * 
     * @param   DateTime    $dt     The DateTime object to format into a UNIX timestamp
     * 
     * @return  int
     * 
     * @static
     * @access  public
     */

    public static function prepareTimestampForInsert(DateTime $dt): int {
        return intval(round(floatval($dt->format('U.u')) * 1000, 0));
    }


    /**
     * {@inheritdoc}
     * 
     * This version of the ActiveRecord automatically builds a set of rules based off of the columns defined in [[columns()]]
     */
    public function rules() {
        $rules = parent::rules();

        // $indexedCols = [];
        // foreach (static::columns() as $name => $type) {
        //     if (!key_exists($type, $indexedCols)) {
        //         $indexedCols[$type] = [$name];
        //     } else {
        //         $indexedCols[$type][] = $name;
        //     }
        // }

        // foreach ($indexedCols as $type => $cols) {
        //     switch ($type) {
        //         case self::PARAM_TAOS_NULL:
        //             $rules[] = [$cols, 'null'];
        //             break;
        //         case self::PARAM_TAOS_BOOL:
        //             $rules[] = [$cols, 'boolean'];
        //             break;
        //         case self::PARAM_TAOS_TINYINT:
        //             // $rules[] = [$cols, 'integer'];
        //             break;
        //         case self::PARAM_TAOS_SMALLINT:
        //             break;
        //         case self::PARAM_TAOS_INT:
        //             $rules[] = [$cols, 'integer'];
        //             break;
        //         case self::PARAM_TAOS_BIGINT:
        //             break;
        //         case self::PARAM_TAOS_FLOAT:
        //             $rules[] = [$cols, 'float'];
        //             break;
        //         case self::PARAM_TAOS_DOUBLE:
        //             $rules[] = [$cols, 'double'];
        //             break;
        //         case self::PARAM_TAOS_VARCHAR:
        //             $rules[] = [$cols, 'string'];
        //             break;
        //         case self::PARAM_TAOS_TIMESTAMP:
        //             $rules[] = [$cols, 'integer'];
        //             break;
        //         case self::PARAM_TAOS_NCHAR:
        //             $rules[] = [$cols, 'string'];
        //             break;
        //         case self::PARAM_TAOS_UTINYINT:
        //             break;
        //         case self::PARAM_TAOS_USMALLINT:
        //             break;
        //         case self::PARAM_TAOS_UINT:
        //             break;
        //         case self::PARAM_TAOS_UBIGINT:
        //             break;
        //         case self::PARAM_TAOS_JSON:
        //             break;
        //         case self::PARAM_TAOS_VARBINARY:
        //             break;
        //         case self::PARAM_TAOS_DECIMAL:
        //             break;
        //         case self::PARAM_TAOS_BLOB:
        //             break;
        //         case self::PARAM_TAOS_MEDIUMBLOB:
        //             break;
        //         case self::PARAM_TAOS_BINARY:
        //             break;
        //         case self::PARAM_TAOS_GEOMETRY:
        //             break;
        //         case self::PARAM_TAOS_MAX:
        //             break;
        //     }
        // }

        return $rules;
    }


    /**
     * {@inheritdoc}
     * @return ActiveQuery the newly created [[Query]] instance.
     */
    public static function find() {
        return Yii::createObject(
            ActiveQuery::class,
            [get_called_class(), [
                'select' => array_keys(static::columns()),
                'from' => static::tableName()
            ]]
        );
    }

    /**
     * {@inheritdoc}
     * @return static|null ActiveRecord instance matching the condition, or `null` if nothing matches.
     */
    public static function findOne($condition) {
        return static::findByCondition($condition)->one();
    }

    /**
     * {@inheritdoc}
     * @return static[] an array of ActiveRecord instances, or an empty array if nothing matches.
     */
    public static function findAll($condition) {
        return static::findByCondition($condition)->all();
    }

    /**
     * Finds ActiveRecord instance(s) by the given condition.
     * This method is internally called by [[findOne()]] and [[findAll()]].
     * @param mixed $condition please refer to [[findOne()]] for the explanation of this parameter
     * @return ActiveQueryInterface the newly created [[ActiveQueryInterface|ActiveQuery]] instance.
     * @throws InvalidConfigException if there is no primary key defined
     * @internal
     */
    protected static function findByCondition($condition) {
        $query = static::find();

        if (!ArrayHelper::isAssociative($condition)) {
            // query by primary key
            $primaryKey = static::primaryKey();
            if (isset($primaryKey[0])) {
                // if condition is scalar, search for a single primary key, if it is array, search for multiple primary key values
                $condition = [$primaryKey[0] => is_array($condition) ? array_values($condition) : $condition];
            } else {
                throw new InvalidConfigException('"' . get_called_class() . '" must have a primary key.');
            }
        }

        return $query->andWhere($condition);
    }




    /**
     * Deletes rows in the table using the provided conditions.
     * WARNING: If you do not specify any condition, this method will delete ALL rows in the table.
     *
     * For example, to delete all customers whose status is 3:
     *
     * ```php
     * Customer::deleteAll('status = 3');
     * ```
     *
     * @param string|array $condition the conditions that will be used to load the models to delete.
     * Please refer to [[Query::where()]] on how to specify this parameter.
     * @return int the number of rows deleted
     * @throws NotSupportedException if not overridden.
     */
    public static function deleteAll($condition = null) {
        $models = static::findAll($condition);
        $rowCount = 0;
        foreach ($models as $model) {
            // delete each of the models
            $command = static::getDb()->createCommand();
            $command->delete(static::tableName(), ['ts' => $model->{self::PRIMARY_TIMESTAMP_NAME}])->execute();
            $rowCount++;
        }
        return $rowCount;
    }


    /**
     * {@inheritdoc}
     */
    public function canGetProperty($name, $checkVars = true, $checkBehaviors = true) {
        if (parent::canGetProperty($name, $checkVars, $checkBehaviors)) {
            return true;
        }

        try {
            return $this->hasAttribute($name);
        } catch (\Exception $e) {
            // `hasAttribute()` may fail on base/abstract classes in case automatic attribute list fetching used
            return false;
        }
    }

    /**
     * {@inheritdoc}
     */
    public function canSetProperty($name, $checkVars = true, $checkBehaviors = true) {
        if (parent::canSetProperty($name, $checkVars, $checkBehaviors)) {
            return true;
        }

        try {
            return $this->hasAttribute($name);
        } catch (\Exception $e) {
            // `hasAttribute()` may fail on base/abstract classes in case automatic attribute list fetching used
            return false;
        }
    }


    /**
     * PHP getter magic method.
     * This method is overridden so that attributes and related objects can be accessed like properties.
     *
     * @param string $name property name
     * @throws InvalidArgumentException if relation name is wrong
     * @return mixed property value
     * @see getAttribute()
     */
    public function __get($name) {
        // if the attribute being accessed is the partition or row key, get that value rather
        if ($name == static::getPrimaryTSName()) {
            $name = self::PRIMARY_TIMESTAMP_NAME;
        }

        if (isset($this->_attributes[$name]) || array_key_exists($name, $this->_attributes)) {
            return $this->getAttributeOfType(ArrayHelper::getValue(static::columns(), $name, self::PARAM_TAOS_NCHAR), $this->_attributes[$name]);
        }

        if ($this->hasAttribute($name)) {
            return null;
        }

        if (isset($this->_related[$name]) || array_key_exists($name, $this->_related)) {
            return $this->_related[$name];
        }
        $value = parent::__get($name);
        if ($value instanceof ActiveQueryInterface) {
            $this->setRelationDependencies($name, $value);
            return $this->_related[$name] = $value->findFor($name, $this);
        }

        return $value;
    }



    /**
     * In order to ease use,
     * @param string $type column type to convert as
     * @param mixed $value
     */
    private function getAttributeOfType(string $type, $value) {
        if ($value === null) {
            return null;
        }
        switch ($type) {
                // case self::DATA_TYPE_BOOLEAN: {
                //         // convert bool to int
                //         if ($value) {
                //             return 1;
                //         } else {
                //             return 0;
                //         }
                //     }
                // case self::DATA_TYPE_DATETIME: {
                //         $value->setTimezone(new DateTimeZone('Africa/Johannesburg'));
                //         return $value->format('Y-m-d H:i:s');
                //     }
                // case self::DATA_TYPE_INT32:
                //     return intval($value);
                // case self::DATA_TYPE_DOUBLE: {
                //         return doubleval($value);
                //     }
                // case self::DATA_TYPE_GUID:
                // case self::DATA_TYPE_BINARY:
                // case self::DATA_TYPE_STRING:
                // case self::DATA_TYPE_INT64:




                // case self::PARAM_TAOS_NULL:
                //     break;
                // case self::PARAM_TAOS_BOOL:
                //     break;
                // case self::PARAM_TAOS_TINYINT:
                //     break;
                // case self::PARAM_TAOS_SMALLINT:
                //     break;
                // case self::PARAM_TAOS_INT:
                //     break;
                // case self::PARAM_TAOS_BIGINT:
                //     break;
                // case self::PARAM_TAOS_FLOAT:
                //     break;
                // case self::PARAM_TAOS_DOUBLE:
                //     break;
                // case self::PARAM_TAOS_BINARY:
                //     break;
                // case self::PARAM_TAOS_TIMESTAMP:
                //     break;
                // case self::PARAM_TAOS_NCHAR:
                //     break;
                // case self::PARAM_TAOS_UTINYINT:
                //     break;
                // case self::PARAM_TAOS_USMALLINT:
                //     break;
                // case self::PARAM_TAOS_UINT:
                //     break;
                // case self::PARAM_TAOS_UBIGINT:
                //     break;
                // case self::PARAM_TAOS_JSON:
                //     break;

            default:
                // either it's a string, or we don't know what it is. Try cast it to string anyhow
                return strval($value);
        }
    }



    /**
     * PHP setter magic method.
     * This method is overridden so that AR attributes can be accessed like properties.
     * @param string $name property name
     * @param mixed $value property value
     */
    public function __set($name, $value) {
        // if the attribute being set is the partition or row key, set that value rather
        if ($name == static::getPrimaryTSName()) {
            $name = self::PRIMARY_TIMESTAMP_NAME;
        }

        if ($this->hasAttribute($name)) {
            if (
                !empty($this->_relationsDependencies[$name])
                && (!array_key_exists($name, $this->_attributes) || $this->_attributes[$name] !== $value)
            ) {
                $this->resetDependentRelations($name);
            }
            $this->setAttributeOfType(ArrayHelper::getValue(static::columns(), $name, self::PARAM_TAOS_NCHAR), $name, $value);
            // $this->_attributes[$name] = $value;
        } else {
            parent::__set($name, $value);
        }
    }



    private function setAttributeOfType(string $type, $name, $value) {
        if ($value === '' && $type != self::PARAM_TAOS_NCHAR) {
            return;
        }
        switch ($type) {
                // case self::DATA_TYPE_BOOLEAN: {
                //         // interpret any truthy value as true, and any falsey as false
                //         if ($value) {
                //             $saveValue = true;
                //         } else {
                //             $saveValue = false;
                //         }
                //         break;
                //     }
                // case self::DATA_TYPE_DATETIME: {
                //         if ($value instanceof DateTime) {
                //             $saveValue = $value;
                //         } else {
                //             $saveValue = new DateTime($value);
                //             $saveValue->setTimezone(new DateTimeZone('UTC'));
                //         }
                //         break;
                //     }
                // case self::DATA_TYPE_INT32:
                //     $saveValue = intval($value);
                //     break;
                // case self::DATA_TYPE_DOUBLE: {
                //         $saveValue = doubleval($value);
                //         break;
                //     }
                // case self::DATA_TYPE_GUID:
                // case self::DATA_TYPE_BINARY:
                // case self::DATA_TYPE_STRING:
                // case self::DATA_TYPE_INT64:


                // case self::PARAM_TAOS_NULL:
                //     break;
                // case self::PARAM_TAOS_BOOL:
                //     break;
                // case self::PARAM_TAOS_TINYINT:
                //     break;
                // case self::PARAM_TAOS_SMALLINT:
                //     break;
                // case self::PARAM_TAOS_INT:
                //     break;
                // case self::PARAM_TAOS_BIGINT:
                //     break;
                // case self::PARAM_TAOS_FLOAT:
                //     break;
                // case self::PARAM_TAOS_DOUBLE:
                //     break;
                // case self::PARAM_TAOS_BINARY:
                //     break;
                // case self::PARAM_TAOS_TIMESTAMP:
                //     break;
                // case self::PARAM_TAOS_NCHAR:
                //     break;
                // case self::PARAM_TAOS_UTINYINT:
                //     break;
                // case self::PARAM_TAOS_USMALLINT:
                //     break;
                // case self::PARAM_TAOS_UINT:
                //     break;
                // case self::PARAM_TAOS_UBIGINT:
                //     break;
                // case self::PARAM_TAOS_JSON:
                //     break;



            default:
                // either it's a string, or we don't know what it is. Try cast it to string anyhow
                $saveValue = strval($value);
                break;
        }
        $this->_attributes[$name] = $saveValue;
    }

    /**
     * Checks if a property value is null.
     * This method overrides the parent implementation by checking if the named attribute is `null` or not.
     * @param string $name the property name or the event name
     * @return bool whether the property value is null
     */
    public function __isset($name) {
        try {
            return $this->__get($name) !== null;
        } catch (\Throwable $t) {
            return false;
        } catch (\Exception $e) {
            return false;
        }
    }

    /**
     * Sets a component property to be null.
     * This method overrides the parent implementation by clearing
     * the specified attribute value.
     * @param string $name the property name or the event name
     */
    public function __unset($name) {
        if ($this->hasAttribute($name)) {
            unset($this->_attributes[$name]);
            if (!empty($this->_relationsDependencies[$name])) {
                $this->resetDependentRelations($name);
            }
        } elseif (array_key_exists($name, $this->_related)) {
            unset($this->_related[$name]);
        } elseif ($this->getRelation($name, false) === null) {
            parent::__unset($name);
        }
    }



    /**
     * Returns the list of all attribute names of the model.
     * The default implementation will return all column names of the table associated with this AR class.
     * @return array list of attribute names.
     */
    public function attributes() {
        $cols = array_keys(static::columns());
        // if the partition key attribute is in the array, remove it
        if (($index = array_search(static::getPrimaryTSName(), $cols)) !== false) {
            // the partition key attribute exists. Kill it
            array_splice($cols, $index, 1);
        }

        // now we also need to add the partitionkey and rowkey values themselves
        $cols[] = self::PRIMARY_TIMESTAMP_NAME;

        return $cols;
    }


    /**
     * Returns a value indicating whether the model has an attribute with the specified name.
     * @param string $name the name of the attribute
     * @return bool whether the model has an attribute with the specified name.
     */
    public function hasAttribute($name) {
        return isset($this->_attributes[$name]) || in_array($name, $this->attributes(), true);
    }

    /**
     * Returns the named attribute value.
     * If this record is the result of a query and the attribute is not loaded,
     * `null` will be returned.
     * @param string $name the attribute name
     * @return mixed the attribute value. `null` if the attribute is not set or does not exist.
     * @see hasAttribute()
     */
    public function getAttribute($name) {
        return isset($this->_attributes[$name]) ? $this->_attributes[$name] : null;
    }


    /**
     * Sets the named attribute value.
     * @param string $name the attribute name
     * @param mixed $value the attribute value.
     * @throws InvalidArgumentException if the named attribute does not exist.
     * @see hasAttribute()
     */

    public function setAttribute($name, $value) {
        if ($this->hasAttribute($name)) {
            if (
                !empty($this->_relationsDependencies[$name])
                && (!array_key_exists($name, $this->_attributes) || $this->_attributes[$name] !== $value)
            ) {
                $this->resetDependentRelations($name);
            }
            $this->_attributes[$name] = $value;
        } else {
            throw new InvalidArgumentException(get_class($this) . ' has no attribute named "' . $name . '".');
        }
    }

    /**
     * Returns the old attribute values.
     * @return array the old attribute values (name-value pairs)
     */
    public function getOldAttributes() {
        return $this->_oldAttributes === null ? [] : $this->_oldAttributes;
    }

    /**
     * Sets the old attribute values.
     * All existing old attribute values will be discarded.
     * @param array|null $values old attribute values to be set.
     * If set to `null` this record is considered to be [[isNewRecord|new]].
     */
    public function setOldAttributes($values) {
        $this->_oldAttributes = $values;
    }




    /**
     * Returns the old value of the named attribute.
     * If this record is the result of a query and the attribute is not loaded,
     * `null` will be returned.
     * @param string $name the attribute name
     * @return mixed the old attribute value. `null` if the attribute is not loaded before
     * or does not exist.
     * @see hasAttribute()
     */
    public function getOldAttribute($name) {
        return isset($this->_oldAttributes[$name]) ? $this->_oldAttributes[$name] : null;
    }

    /**
     * Sets the old value of the named attribute.
     * @param string $name the attribute name
     * @param mixed $value the old attribute value.
     * @throws InvalidArgumentException if the named attribute does not exist.
     * @see hasAttribute()
     */
    public function setOldAttribute($name, $value) {
        if (isset($this->_oldAttributes[$name]) || $this->hasAttribute($name)) {
            $this->_oldAttributes[$name] = $value;
        } else {
            throw new InvalidArgumentException(get_class($this) . ' has no attribute named "' . $name . '".');
        }
    }


    /**
     * Marks an attribute dirty.
     * This method may be called to force updating a record when calling [[update()]],
     * even if there is no change being made to the record.
     * @param string $name the attribute name
     */
    public function markAttributeDirty($name) {
        unset($this->_oldAttributes[$name]);
    }

    /**
     * Returns a value indicating whether the named attribute has been changed.
     * @param string $name the name of the attribute.
     * @param bool $identical whether the comparison of new and old value is made for
     * identical values using `===`, defaults to `true`. Otherwise `==` is used for comparison.
     * This parameter is available since version 2.0.4.
     * @return bool whether the attribute has been changed
     */
    public function isAttributeChanged($name, $identical = true) {
        if (isset($this->_attributes[$name], $this->_oldAttributes[$name])) {
            if ($identical) {
                return $this->_attributes[$name] !== $this->_oldAttributes[$name];
            }

            return $this->_attributes[$name] != $this->_oldAttributes[$name];
        }

        return isset($this->_attributes[$name]) || isset($this->_oldAttributes[$name]);
    }



    /**
     * Returns the attribute values that have been modified since they are loaded or saved most recently.
     *
     * The comparison of new and old values is made for identical values using `===`.
     *
     * @param string[]|null $names the names of the attributes whose values may be returned if they are
     * changed recently. If null, [[attributes()]] will be used.
     * @return array the changed attribute values (name-value pairs)
     */
    public function getDirtyAttributes($names = null) {
        if ($names === null) {
            $names = $this->attributes();
        }
        $names = array_flip($names);
        $attributes = [];
        if ($this->_oldAttributes === null) {
            foreach ($this->_attributes as $name => $value) {
                if (isset($names[$name])) {
                    $attributes[$name] = $value;
                }
            }
        } else {
            foreach ($this->_attributes as $name => $value) {
                if (isset($names[$name]) && (!array_key_exists($name, $this->_oldAttributes) || $value !== $this->_oldAttributes[$name])) {
                    $attributes[$name] = $value;
                }
            }
        }

        return $attributes;
    }



    /**
     * Saves the current record.
     *
     * This method will call [[insert()]] when [[isNewRecord]] is `true`, or [[update()]]
     * when [[isNewRecord]] is `false`.
     *
     * For example, to save a customer record:
     *
     * ```php
     * $customer = new Customer; // or $customer = Customer::findOne($id);
     * $customer->name = $name;
     * $customer->email = $email;
     * $customer->save();
     * ```
     *
     * @param bool $runValidation whether to perform validation (calling [[validate()]])
     * before saving the record. Defaults to `true`. If the validation fails, the record
     * will not be saved to the database and this method will return `false`.
     * @param array $attributeNames list of attribute names that need to be saved. Defaults to null,
     * meaning all attributes that are loaded from DB will be saved.
     * @return bool whether the saving succeeded (i.e. no validation errors occurred).
     */
    public function save($runValidation = true, $attributeNames = null) {
        if ($this->getIsNewRecord()) {
            return $this->insert($runValidation, $attributeNames);
        }

        return $this->update($runValidation, $attributeNames) !== false;
    }



    /**
     * Inserts a row into the associated database table using the attribute values of this record.
     *
     * This method performs the following steps in order:
     *
     * 1. call [[beforeValidate()]] when `$runValidation` is `true`. If [[beforeValidate()]]
     *    returns `false`, the rest of the steps will be skipped;
     * 2. call [[afterValidate()]] when `$runValidation` is `true`. If validation
     *    failed, the rest of the steps will be skipped;
     * 3. call [[beforeSave()]]. If [[beforeSave()]] returns `false`,
     *    the rest of the steps will be skipped;
     * 4. insert the record into database. If this fails, it will skip the rest of the steps;
     * 5. call [[afterSave()]];
     *
     * In the above step 1, 2, 3 and 5, events [[EVENT_BEFORE_VALIDATE]],
     * [[EVENT_AFTER_VALIDATE]], [[EVENT_BEFORE_INSERT]], and [[EVENT_AFTER_INSERT]]
     * will be raised by the corresponding methods.
     *
     * Only the [[dirtyAttributes|changed attribute values]] will be inserted into database.
     *
     * If the table's primary key is auto-incremental and is `null` during insertion,
     * it will be populated with the actual value after insertion.
     *
     * For example, to insert a customer record:
     *
     * ```php
     * $customer = new Customer;
     * $customer->name = $name;
     * $customer->email = $email;
     * $customer->insert();
     * ```
     *
     * @param bool $runValidation whether to perform validation (calling [[validate()]])
     * before saving the record. Defaults to `true`. If the validation fails, the record
     * will not be saved to the database and this method will return `false`.
     * @param array $attributes list of attributes that need to be saved. Defaults to `null`,
     * meaning all attributes that are loaded from DB will be saved.
     * @return bool whether the attributes are valid and the record is inserted successfully.
     * @throws \Exception|\Throwable in case insert failed.
     */
    public function insert($runValidation = true, $attributes = null) {
        if ($runValidation && !$this->validate($attributes)) {
            Yii::info('Model not inserted due to validation error.', __METHOD__);
            return false;
        }

        try {
            $result = $this->insertInternal($attributes);

            return $result;
        } catch (\Exception $e) {
            throw $e;
        } catch (\Throwable $e) {
            throw $e;
        }
    }





    /**
     * Inserts an ActiveRecord into DB without considering transaction.
     * @param array $attributes list of attributes that need to be saved. Defaults to `null`,
     * meaning all attributes that are loaded from DB will be saved.
     * @return bool whether the record is inserted successfully.
     */
    protected function insertInternal($attributes = null) {
        if (!$this->beforeSave(true)) {
            return false;
        }
        $values = $this->getDirtyAttributes($attributes);
        $columns = static::columns();
        $preppedColumns = [];
        foreach ($values as $name => $value) {
            $preppedColumns[$name] = [
                'value' => $value,
                'type' => ArrayHelper::getValue($columns, $name, self::PARAM_TAOS_NCHAR)
            ];
        }

        if (($primaryKeys = static::getDb()->createCommand()->insert(static::tableName(), $preppedColumns)->execute()) === false) {
            return false;
        }

        $changedAttributes = array_fill_keys(array_keys($values), null);
        $this->setOldAttributes($values);
        $this->afterSave(true, $changedAttributes);

        return true;
    }



    /**
     * Deletes the table row corresponding to this active record.
     *
     * This method performs the following steps in order:
     *
     * 1. call [[beforeDelete()]]. If the method returns `false`, it will skip the
     *    rest of the steps;
     * 2. delete the record from the database;
     * 3. call [[afterDelete()]].
     *
     * In the above step 1 and 3, events named [[EVENT_BEFORE_DELETE]] and [[EVENT_AFTER_DELETE]]
     * will be raised by the corresponding methods.
     *
     * @return int|false the number of rows deleted, or `false` if the deletion is unsuccessful for some reason.
     * Note that it is possible the number of rows deleted is 0, even though the deletion execution is successful.
     * @throws StaleObjectException if [[optimisticLock|optimistic locking]] is enabled and the data
     * being deleted is outdated.
     * @throws Exception in case delete failed.
     */
    public function delete() {
        $result = false;
        if ($this->beforeDelete()) {
            // we do not check the return value of deleteAll() because it's possible
            // the record is already deleted in the database and thus the method will return 0
            $condition = $this->getOldPrimaryKey(true);
            $lock = $this->optimisticLock();
            if ($lock !== null) {
                $condition[$lock] = $this->$lock;
            }
            $result = static::deleteAll($condition);
            if ($lock !== null && !$result) {
                throw new StaleObjectException('The object being deleted is outdated.');
            }
            $this->_oldAttributes = null;
            $this->afterDelete();
        }

        return $result;
    }

    /**
     * Returns a value indicating whether the current record is new.
     * @return bool whether the record is new and should be inserted when calling [[save()]].
     */
    public function getIsNewRecord() {
        return $this->_oldAttributes === null;
    }


    /**
     * Sets the value indicating whether the record is new.
     * @param bool $value whether the record is new and should be inserted when calling [[save()]].
     * @see getIsNewRecord()
     */
    public function setIsNewRecord($value) {
        $this->_oldAttributes = $value ? null : $this->_attributes;
    }

    /**
     * Initializes the object.
     * This method is called at the end of the constructor.
     * The default implementation will trigger an [[EVENT_INIT]] event.
     */
    public function init() {
        parent::init();
        $this->trigger(self::EVENT_INIT);
    }


    /**
     * This method is called when the AR object is created and populated with the query result.
     * The default implementation will trigger an [[EVENT_AFTER_FIND]] event.
     * When overriding this method, make sure you call the parent implementation to ensure the
     * event is triggered.
     */
    public function afterFind() {
        $this->trigger(self::EVENT_AFTER_FIND);
    }


    /**
     * This method is called at the beginning of inserting or updating a record.
     *
     * The default implementation will trigger an [[EVENT_BEFORE_INSERT]] event when `$insert` is `true`,
     * or an [[EVENT_BEFORE_UPDATE]] event if `$insert` is `false`.
     * When overriding this method, make sure you call the parent implementation like the following:
     *
     * ```php
     * public function beforeSave($insert)
     * {
     *     if (!parent::beforeSave($insert)) {
     *         return false;
     *     }
     *
     *     // ...custom code here...
     *     return true;
     * }
     * ```
     *
     * @param bool $insert whether this method called while inserting a record.
     * If `false`, it means the method is called while updating a record.
     * @return bool whether the insertion or updating should continue.
     * If `false`, the insertion or updating will be cancelled.
     */
    public function beforeSave($insert) {
        $event = new ModelEvent();
        $this->trigger($insert ? self::EVENT_BEFORE_INSERT : self::EVENT_BEFORE_UPDATE, $event);

        return $event->isValid;
    }


    /**
     * This method is called at the end of inserting or updating a record.
     * The default implementation will trigger an [[EVENT_AFTER_INSERT]] event when `$insert` is `true`,
     * or an [[EVENT_AFTER_UPDATE]] event if `$insert` is `false`. The event class used is [[AfterSaveEvent]].
     * When overriding this method, make sure you call the parent implementation so that
     * the event is triggered.
     * @param bool $insert whether this method called while inserting a record.
     * If `false`, it means the method is called while updating a record.
     * @param array $changedAttributes The old values of attributes that had changed and were saved.
     * You can use this parameter to take action based on the changes made for example send an email
     * when the password had changed or implement audit trail that tracks all the changes.
     * `$changedAttributes` gives you the old attribute values while the active record (`$this`) has
     * already the new, updated values.
     *
     * Note that no automatic type conversion performed by default. You may use
     * [[\yii\behaviors\AttributeTypecastBehavior]] to facilitate attribute typecasting.
     * See http://www.yiiframework.com/doc-2.0/guide-db-active-record.html#attributes-typecasting.
     */
    public function afterSave($insert, $changedAttributes) {
        $this->trigger($insert ? self::EVENT_AFTER_INSERT : self::EVENT_AFTER_UPDATE, new AfterSaveEvent([
            'changedAttributes' => $changedAttributes,
        ]));
    }



    /**
     * This method is invoked before deleting a record.
     *
     * The default implementation raises the [[EVENT_BEFORE_DELETE]] event.
     * When overriding this method, make sure you call the parent implementation like the following:
     *
     * ```php
     * public function beforeDelete()
     * {
     *     if (!parent::beforeDelete()) {
     *         return false;
     *     }
     *
     *     // ...custom code here...
     *     return true;
     * }
     * ```
     *
     * @return bool whether the record should be deleted. Defaults to `true`.
     */
    public function beforeDelete() {
        $event = new ModelEvent();
        $this->trigger(self::EVENT_BEFORE_DELETE, $event);

        return $event->isValid;
    }


    /**
     * This method is invoked after deleting a record.
     * The default implementation raises the [[EVENT_AFTER_DELETE]] event.
     * You may override this method to do postprocessing after the record is deleted.
     * Make sure you call the parent implementation so that the event is raised properly.
     */
    public function afterDelete() {
        $this->trigger(self::EVENT_AFTER_DELETE);
    }

    /**
     * Repopulates this active record with the latest data.
     *
     * If the refresh is successful, an [[EVENT_AFTER_REFRESH]] event will be triggered.
     * This event is available since version 2.0.8.
     *
     * @return bool whether the row still exists in the database. If `true`, the latest data
     * will be populated to this active record. Otherwise, this record will remain unchanged.
     */
    public function refresh() {
        /* @var $record BaseActiveRecord */
        $record = static::findOne($this->getPrimaryKey(true));
        return $this->refreshInternal($record);
    }


    /**
     * Repopulates this active record with the latest data from a newly fetched instance.
     * @param BaseActiveRecord $record the record to take attributes from.
     * @return bool whether refresh was successful.
     * @see refresh()
     * @since 2.0.13
     */
    protected function refreshInternal($record) {
        if ($record === null) {
            return false;
        }
        foreach ($this->attributes() as $name) {
            $this->_attributes[$name] = isset($record->_attributes[$name]) ? $record->_attributes[$name] : null;
        }
        $this->_oldAttributes = $record->_oldAttributes;
        $this->_related = [];
        $this->_relationsDependencies = [];
        $this->afterRefresh();

        return true;
    }


    /**
     * This method is called when the AR object is refreshed.
     * The default implementation will trigger an [[EVENT_AFTER_REFRESH]] event.
     * When overriding this method, make sure you call the parent implementation to ensure the
     * event is triggered.
     * @since 2.0.8
     */
    public function afterRefresh() {
        $this->trigger(self::EVENT_AFTER_REFRESH);
    }


    /**
     * Returns a value indicating whether the given active record is the same as the current one.
     * The comparison is made by comparing the table names and the primary key values of the two active records.
     * If one of the records [[isNewRecord|is new]] they are also considered not equal.
     * @param ActiveRecordInterface $record record to compare to
     * @return bool whether the two active records refer to the same row in the same database table.
     */
    public function equals($record) {
        if ($this->getIsNewRecord() || $record->getIsNewRecord()) {
            return false;
        }

        return get_class($this) === get_class($record) && $this->getPrimaryKey() === $record->getPrimaryKey();
    }


    /**
     * Returns the primary key name(s) for this AR class.
     * The default implementation will return the primary key(s) as declared
     * in the DB table that is associated with this AR class.
     *
     * If the DB table does not declare any primary key, you should override
     * this method to return the attributes that you want to use as primary keys
     * for this AR class.
     *
     * Note that an array should be returned even for a table with single primary key.
     *
     * @return string[] the primary keys of the associated database table.
     */
    public static function primaryKey() {
        return ['ts'];
    }



    /**
     * Returns the primary key value(s).
     * @param bool $asArray whether to return the primary key value as an array. If `true`,
     * the return value will be an array with column names as keys and column values as values.
     * Note that for composite primary keys, an array will always be returned regardless of this parameter value.
     * @property mixed The primary key value. An array (column name => column value) is returned if
     * the primary key is composite. A string is returned otherwise (null will be returned if
     * the key value is null).
     * @return mixed the primary key value. An array (column name => column value) is returned if the primary key
     * is composite or `$asArray` is `true`. A string is returned otherwise (null will be returned if
     * the key value is null).
     */
    public function getPrimaryKey($asArray = false) {
        return [$this->_attributes['ts']];
    }


    /**
     * Returns the old primary key value(s).
     * This refers to the primary key value that is populated into the record
     * after executing a find method (e.g. find(), findOne()).
     * The value remains unchanged even if the primary key attribute is manually assigned with a different value.
     * @param bool $asArray whether to return the primary key value as an array. If `true`,
     * the return value will be an array with column name as key and column value as value.
     * If this is `false` (default), a scalar value will be returned for non-composite primary key.
     * @property mixed The old primary key value. An array (column name => column value) is
     * returned if the primary key is composite. A string is returned otherwise (null will be
     * returned if the key value is null).
     * @return mixed the old primary key value. An array (column name => column value) is returned if the primary key
     * is composite or `$asArray` is `true`. A string is returned otherwise (null will be returned if
     * the key value is null).
     * @throws Exception if the AR model does not have a primary key
     */
    public function getOldPrimaryKey($asArray = false) {
        $keys = $this->primaryKey();
        if (empty($keys)) {
            throw new Exception(get_class($this) . ' does not have a primary key. You should either define a primary key for the corresponding table or override the primaryKey() method.');
        }
        if (!$asArray && count($keys) === 1) {
            return isset($this->_oldAttributes[$keys[0]]) ? $this->_oldAttributes[$keys[0]] : null;
        }

        $values = [];
        foreach ($keys as $name) {
            $values[$name] = isset($this->_oldAttributes[$name]) ? $this->_oldAttributes[$name] : null;
        }

        return $values;
    }


    /**
     * Populates an active record object using a row of data from the database/storage.
     *
     * This is an internal method meant to be called to create active record objects after
     * fetching data from the database. It is mainly used by [[ActiveQuery]] to populate
     * the query results into active records.
     *
     * When calling this method manually you should call [[afterFind()]] on the created
     * record to trigger the [[EVENT_AFTER_FIND|afterFind Event]].
     *
     * @param BaseActiveRecord $record the record to be populated. In most cases this will be an instance
     * created by [[instantiate()]] beforehand.
     * @param array $row attribute values (name => value)
     */
    public static function populateRecord($record, $row) {
        $columns = array_flip($record->attributes());
        foreach ($row as $name => $value) {
            if (isset($columns[$name])) {
                $record->_attributes[$name] = $value;
            } elseif ($record->canSetProperty($name)) {
                $record->$name = $value;
            }
        }
        $record->_oldAttributes = $record->_attributes;
        $record->_related = [];
        $record->_relationsDependencies = [];
    }



    /**
     * Creates an active record instance.
     *
     * This method is called together with [[populateRecord()]] by [[ActiveQuery]].
     * It is not meant to be used for creating new records directly.
     *
     * You may override this method if the instance being created
     * depends on the row data to be populated into the record.
     * For example, by creating a record based on the value of a column,
     * you may implement the so-called single-table inheritance mapping.
     * @param array $row row data to be populated into the record.
     * @return static the newly created active record
     */
    public static function instantiate($row) {
        return new static();
    }

    /**
     * Returns whether there is an element at the specified offset.
     * This method is required by the interface [[\ArrayAccess]].
     * @param mixed $offset the offset to check on
     * @return bool whether there is an element at the specified offset.
     */
    public function offsetExists($offset) {
        return $this->__isset($offset);
    }


    /**
     * Returns a value indicating whether the given set of attributes represents the primary key for this model.
     * @param array $keys the set of attributes to check
     * @return bool whether the given set of attributes represents the primary key for this model
     */
    public static function isPrimaryKey($keys) {
        $pks = static::primaryKey();
        if (count($keys) === count($pks)) {
            return count(array_intersect($keys, $pks)) === count($pks);
        }

        return false;
    }


    /**
     * Returns the text label for the specified attribute.
     * If the attribute looks like `relatedModel.attribute`, then the attribute will be received from the related model.
     * @param string $attribute the attribute name
     * @return string the attribute label
     * @see generateAttributeLabel()
     * @see attributeLabels()
     */
    public function getAttributeLabel($attribute) {
        $labels = $this->attributeLabels();
        if (isset($labels[$attribute])) {
            return $labels[$attribute];
        } elseif (strpos($attribute, '.')) {
            $attributeParts = explode('.', $attribute);
            $neededAttribute = array_pop($attributeParts);

            $relatedModel = $this;
            foreach ($attributeParts as $relationName) {
                if ($relatedModel->isRelationPopulated($relationName) && $relatedModel->$relationName instanceof self) {
                    $relatedModel = $relatedModel->$relationName;
                } else {
                    try {
                        $relation = $relatedModel->getRelation($relationName);
                    } catch (InvalidArgumentException $e) {
                        return $this->generateAttributeLabel($attribute);
                    }
                    /* @var $modelClass ActiveRecordInterface */
                    $modelClass = $relation->modelClass;
                    $relatedModel = $modelClass::instance();
                }
            }

            $labels = $relatedModel->attributeLabels();
            if (isset($labels[$neededAttribute])) {
                return $labels[$neededAttribute];
            }
        }

        return $this->generateAttributeLabel($attribute);
    }


    /**
     * Returns the text hint for the specified attribute.
     * If the attribute looks like `relatedModel.attribute`, then the attribute will be received from the related model.
     * @param string $attribute the attribute name
     * @return string the attribute hint
     * @see attributeHints()
     * @since 2.0.4
     */
    public function getAttributeHint($attribute) {
        $hints = $this->attributeHints();
        if (isset($hints[$attribute])) {
            return $hints[$attribute];
        } elseif (strpos($attribute, '.')) {
            $attributeParts = explode('.', $attribute);
            $neededAttribute = array_pop($attributeParts);

            $relatedModel = $this;
            foreach ($attributeParts as $relationName) {
                if ($relatedModel->isRelationPopulated($relationName) && $relatedModel->$relationName instanceof self) {
                    $relatedModel = $relatedModel->$relationName;
                } else {
                    try {
                        $relation = $relatedModel->getRelation($relationName);
                    } catch (InvalidParamException $e) {
                        return '';
                    }
                    /* @var $modelClass ActiveRecordInterface */
                    $modelClass = $relation->modelClass;
                    $relatedModel = $modelClass::instance();
                }
            }

            $hints = $relatedModel->attributeHints();
            if (isset($hints[$neededAttribute])) {
                return $hints[$neededAttribute];
            }
        }

        return '';
    }


    /**
     * {@inheritdoc}
     *
     * The default implementation returns the names of the columns whose values have been populated into this record.
     */
    public function fields() {
        $fields = array_keys($this->_attributes);

        return array_combine($fields, $fields);
    }




    /**
     * Sets the element value at the specified offset to null.
     * This method is required by the SPL interface [[\ArrayAccess]].
     * It is implicitly called when you use something like `unset($model[$offset])`.
     * @param mixed $offset the offset to unset element
     */
    public function offsetUnset($offset) {
        if (property_exists($this, $offset)) {
            $this->$offset = null;
        } else {
            unset($this->$offset);
        }
    }

    /**
     * Resets dependent related models checking if their links contain specific attribute.
     * @param string $attribute The changed attribute name.
     */
    private function resetDependentRelations($attribute) {
        foreach ($this->_relationsDependencies[$attribute] as $relation) {
            unset($this->_related[$relation]);
        }
        unset($this->_relationsDependencies[$attribute]);
    }

    /**
     * Sets relation dependencies for a property
     * @param string $name property name
     * @param ActiveQueryInterface $relation relation instance
     * @param string|null $viaRelationName intermediate relation
     */
    private function setRelationDependencies($name, $relation, $viaRelationName = null) {
        if (empty($relation->via) && $relation->link) {
            foreach ($relation->link as $attribute) {
                $this->_relationsDependencies[$attribute][$name] = $name;
                if ($viaRelationName !== null) {
                    $this->_relationsDependencies[$attribute][] = $viaRelationName;
                }
            }
        } elseif ($relation->via instanceof ActiveQueryInterface) {
            $this->setRelationDependencies($name, $relation->via);
        } elseif (is_array($relation->via)) {
            list($viaRelationName, $viaQuery) = $relation->via;
            $this->setRelationDependencies($name, $viaQuery, $viaRelationName);
        }
    }
}
