<?php

namespace onespace\tutorial\widgets;

use onespace\tutorial\models\Tutorial as TutorialModel;
use Yii;
use yii\bootstrap\Html;
use yii\helpers\ArrayHelper;
use yii\helpers\Json;
use yii\helpers\Url;
use yii\web\JsExpression;
use yii\web\View;

/**
 * This widget is used to generate DriverJs items
 * 
 * @see https://driverjs.com/docs/basic-usage
 * 
 * @author  Jason Phillips <jason@one-space.co.za>
 */
class Tutorial extends DriverWidget {

    /** @var string path to look for in the database. Defaults to Yii path. Ignored if $steps is defined */
    public string $pagePath;
    /** @var string if this is a specific update version. Used to filter from the db. Defaults to blank string. Ignored if $steps is defined */
    public string $updateVersion = '';

    /**
     * @var array the options for rendering the toggle button tag.
     * The toggle button is used to launch the tutorial.
     * If this property is false, no toggle button will be rendered.
     *
     * The following special options are supported:
     *
     * - tag: string, the tag name of the button. Defaults to 'button'.
     * - label: string, the label of the button. Defaults to 'Show'.
     *
     * The rest of the options will be rendered as the HTML attributes of the button tag.
     * Please refer to the [Modal plugin help](http://getbootstrap.com/javascript/#modals)
     * for the supported HTML attributes.
     */
    public array|bool $launchButton = [];

    /** @var bool Whether to launch the tutorial automatically on page load or not */
    public bool $autoLaunch = false;

    /** @var array Options to pass to the underlying plugin. See https://driverjs.com/docs/configuration for documentation */
    public array $pluginOptions = [];

    /** @var array Manually defined set of steps. Need to be structured according to what the plugin accepts. Will override $pagePath */
    public array $steps = [];

    /** @var string Name of the output Javascript config object, which can be referenced outside of the widget as required */
    public string $jsObj = '';
    /** @var string Name of the output Javascript config object, which can be referenced outside of the widget as required */
    public string $jsMethod = '';

    /** @var bool Whether or not to track when users complete this tutorial. Note, does not work with manually-defined steps */
    public bool $trackCompletion = true;
    public string $completionUrl = '';

    /**
     * {@inheritdoc}
     */
     public function init(): void {
        parent::init();

        // default the page path
        $this->pagePath ??= Yii::$app->controller->action->getUniqueId();
        // if the user hasn't set the js variable name
        if ($this->jsObj == null) {
            $this->jsObj = 'driverObj'.$this->getId(true);
        }
        if ($this->jsMethod == null) {
            $this->jsMethod = 'launchDriver'.$this->getId(true);
        }
        $this->view->registerJs(<<<JS
            currentDriver = null;
        JS, View::POS_BEGIN, self::class.'.basic');

        if ($this->completionUrl == null) {
            $this->completionUrl = Url::toRoute('/tutorial/widget/complete');
        }

        // set the defaults
        if ($this->launchButton !== false) {
            $this->launchButton = array_merge([
                'id' => $this->getId(true),
                'onClick' => $this->jsMethod.'();',
            ], $this->launchButton);
        }

        // if the page path is set, load directly from the db
        if ($this->steps == null) {
            // load from DB
            $stepQuery = TutorialModel::find();
            if ($this->updateVersion == null) {
                $stepQuery->pageTutorial($this->pagePath);
            } else {
                $stepQuery->updateTutorial($this->pagePath, $this->updateVersion);
            }
            $stepQuery->inOrder();
            foreach ($stepQuery->each() as $step) {
                /** @var TutorialModel $step */
                $stepArr = [];
                if ($step->highlight_element != null) {
                    $stepArr['element'] = $step->highlight_element;
                }
                if ($step->title != null) {
                    $stepArr['popover'] ?? [];
                    $stepArr['popover']['title'] = $step->title;
                }
                if ($step->content != null) {
                    $stepArr['popover'] ?? [];
                    $stepArr['popover']['description'] = $step->content;
                }
                if ($step->side != null) {
                    $stepArr['popover'] ?? [];
                    $stepArr['popover']['side'] = $step->side;
                }
                if ($step->align != null) {
                    $stepArr['popover'] ?? [];
                    $stepArr['popover']['align'] = $step->align;
                }
                $this->steps[] = $stepArr;
            }
        } else {
            // we can't track completion on a manual tutorial
            $this->trackCompletion = false;
        }

        // render the settings into JS
        $defaultSettings = [
            'steps' => $this->steps 
        ];
        if ($this->trackCompletion) {
            $defaultSettings['onDestroyStarted'] = new JsExpression(<<<JS
                (element, step, options) => {
                    console.log(element);
                    console.log(step);
                    console.log(options);
                    if (currentDriver != null) {
                        if (!currentDriver.hasNextStep()) {
                            // notify completion
                            $.post({
                                url: '{$this->completionUrl}', // your controller action
                                data: {
                                    pagePath: '{$this->pagePath}',
                                    updateVersion: '{$this->updateVersion}'
                                },
                                type: 'POST',
                                success: function(data) {
                                    console.log('Logged completion');
                                },
                            });
                        }
                        currentDriver.destroy();
                    }
                }
            JS);
        }
        $jsSettings = ArrayHelper::merge($defaultSettings, $this->pluginOptions);
        $jsSettingsString = Json::encode($jsSettings);

        // define the variable we're going to be using at a high enough level to access it
        $this->view->registerJs(<<<JS
            {$this->jsObj} = $jsSettingsString;
            function {$this->jsMethod}() {
                currentDriver = window.driver.js.driver({$this->jsObj});
                currentDriver.drive();
            }
        JS, View::POS_BEGIN);
    }


    /**
     * {@inheritdoc}
     * 
     * @return  string
     */

     public function run(): string {
        // if there are no steps, nothing to see here
        if ($this->steps == null) {
            return '';
        }

        // auto launch if configured to do so
        if ($this->autoLaunch) {
            $this->view->registerJs(<<<JS
                {$this->jsMethod}();
            JS, View::POS_READY);
        }
        
        // if there's a definition to the button, render it
        if (($launchButton = $this->launchButton) !== false) {
            $tag = ArrayHelper::remove($launchButton, 'tag', 'button');
            $label = ArrayHelper::remove($launchButton, 'label', 'Help');
            if ($tag === 'button' && !isset($launchButton['type'])) {
                $launchButton['type'] = 'button';
            }

            return Html::tag($tag, $label, $launchButton);
        } else {
            return '';
        }
    }
}