<?php

namespace onespace\inertia\web;

use stdClass;
use Yii;
use yii\helpers\ArrayHelper;
use yii\rbac\Role;

abstract class Controller extends \yii\web\Controller {
    /**
     * Whether or not to use the controller ID when defining the front end view file.
     *
     * Default is `true`.
     *
     * Overwrite in the child class if you with to define the value for the entire controller, otherwise set
     * in the method for a once off value change.
     */
    protected bool $useControllerId = true;

    /**
     * {@inheritdoc}
     */
    public function init() {
        parent::init();
        $this->layout = $this->setLayoutFile();
        $this->setPageTitle(Yii::$app->params['defaultPageTitle']);
    }

    /**
     * Renders an inertia page. This basically fetches the data via ajax and appends it to the root element.
     * 
     * @param string $component
     * @param array $params
     * 
     * @return array|string
     * @throws \yii\base\InvalidConfigException
     */
    public function inertia(string $component, array $params = []): array|string {
        if ($this->useControllerId) {
            $component = Yii::$app->controller->id . '/' . ltrim($component, " \n\r\t\v\0/");
        }

        $allRoles = [];
        try {
            $authManager = Yii::$app->authManager;
            $roles = $authManager->getRolesByUser(Yii::$app->user->id);

            foreach ($roles as $role) {
                $this->addRoleWithChildren($role, $allRoles, $authManager);
            }
        } catch (\Throwable $th) {
        }
        $finalRoleList = array_values(array_unique($allRoles));

        $pageParams = [
            'component' => $this->correctNameConvention($component),
            'props' => ArrayHelper::merge(
                $this->getInertiaProps($params),
                [
                    'user' => [
                        'identity' => ArrayHelper::toArray(Yii::$app->user->identity ?? new stdClass),
                        'profile' => ArrayHelper::toArray(Yii::$app->user->identity->profile ?? new stdClass),
                        'roles' => $finalRoleList,
                        // 'profileImg' => Yii::$app->params['serviceUrls']['userUi'] ?? . 'avatar/' . Yii::$app->user->identity->dynamics_guid,
                    ],
                    'params' => Yii::$app->params ?? [],
                    'toasts' => Yii::$app->toast->getAllMessages() ?? [],
                    'flashes' => Yii::$app->flash->getAllMessages() ?? [],
                    'title' => Yii::$app->view->title,
                ]
            ),
            'url' => $this->getInertiaUrl(),
            'version' => $this->getInertiaVersion(),
        ];

        if (Yii::$app->request->headers->has('X-Inertia')) {
            return $pageParams;
        }

        $view = Yii::$app->get('inertia')->view;
        return $this->render($view, [
            'page' => $pageParams,
        ]);
    }

    /**
     * Overwrite this in the controller if a different view file is required.
     *
     * This file will be (by default) as defined here: `views/layout/inertia.php`
     */
    protected function setLayoutFile(): string {
        return 'inertia';
    }

    /**
     * Set the page title in a single method call.
     *
     * @param   string  $title  The desired title.
     *
     * @access  protected
     */
    protected function setPageTitle(string $title): void {
        $this->getView()->title = $title;
    }

    private function correctNameConvention(string $component): string {
        $parts = explode('/', $component);
        $parts[array_key_last($parts)] = ucfirst($parts[array_key_last($parts)]);
        return implode('/', $parts);
    }

    /**
     * @param array $params
     * @return array
     * @throws \yii\base\InvalidConfigException
     */
    private function getInertiaProps($params = []) {
        return array_merge(
            Yii::$app->get('inertia')->getShared(),
            $params
        );
    }

    /**
     * @return string
     * @throws \yii\base\InvalidConfigException
     */
    private function getInertiaUrl() {
        $url = Yii::$app->request->getUrl();
        return $url;
    }

    /**
     * @return mixed
     * @throws \yii\base\InvalidConfigException
     */
    private function getInertiaVersion() {
        return Yii::$app->get('inertia')->getVersion();
    }


    private function addRoleWithChildren(Role $role, array &$allRoles, $authManager) {
        $allRoles[] = $role->name;
        $children = $authManager->getChildren($role->name);

        foreach ($children as $child) {
            if ($child instanceof Role) {
                $this->addRoleWithChildren($child, $allRoles, $authManager);
            }
        }
    }
}
