import { DataTable, DataTableProps, DataTableValueArray } from "primereact/datatable";
import { OSIcon } from "../icon";
import { Children, cloneElement, useCallback, useRef } from "react";
import { Column, ColumnProps } from "primereact/column";
import { Skeleton } from "primereact/skeleton";
import { Maximize, MaximizeArea, MaximizeButton } from "../maximize";
import { OSLabel } from "../label";
import { classNames } from "primereact/utils";

type Props<TValue extends DataTableValueArray> = DataTableProps<TValue> & {
    //Values that aren't used but should be accepted to prevent DOM errors when exploding data provider.
    rawData?: any;
    totalPages?: number | undefined;
}

export function OSDataTable<TValue extends DataTableValueArray>({ rawData, className, totalPages, children, loading, ...props }: Props<TValue>) {

    const maximize = useRef<Maximize>(null);

    //Handle custom skeleton-based loading animation
    let isSkeleton: boolean = false;
    if (loading && props.rows) {
        isSkeleton = true;
        let data: DataTableValueArray = [];
        for (let index = 0; index < props.rows; index++) {
            data.push({});
        }
        props.value = data as TValue;
    }


    //Configure defaults for Columns - no other solution seemed to work (HOC, passthrough etc.)
    const modifiedChildren = Children.map(children, (child: any) => {
        if (child?.type?.name === Column.name) {
            return cloneElement<ColumnProps>(child, {
                showFilterMenu: child.props.showFilterMenu ?? false,
                body: isSkeleton ? <Skeleton height="2rem" /> : child.props.body
            });
        }
        return child;
    });


    const getHeaderText = useCallback(() => {
        const { first = 0, rows = 0, totalRecords = 0 } = props;
        let last = (Math.floor(first / rows) + 1) * rows;
        if (last > totalRecords) {
            last = totalRecords;
        }
        if (loading) return "";
        if (last === 0) return ""; //Table itself shows 'No Records found', so no need for duplication.
        if (props.paginator) return `Showing ${first + 1} to ${last} out of ${totalRecords} total records`;
        return `${totalRecords} records found`;
    }, [props.first, props.rows, props.totalRecords, loading, props.paginator]);


    return (<div className="flex flex-col">
        <div className="flex relative pl-2">
            {!loading && <OSLabel label={getHeaderText()} tag={"small"} />}
            <div className="grow" />
            <MaximizeButton maximizeRef={maximize} />
        </div>
        <MaximizeArea ref={maximize}>
            <DataTable
                className={classNames("os-data-table", className)}
                loading={false}
                filterDisplay="row"
                scrollable
                stripedRows
                children={modifiedChildren}
                filterClearIcon={
                    <OSIcon
                        identifier="filter-circle-xmark"
                        colour="var(--colour-danger-red)"
                        tooltip="Clear filter"
                        size={"lg"}
                    />
                }
                sortIcon={(args) => {
                    return (
                        <a
                            className={
                                args.sortOrder === 1
                                    ? "asc"
                                    : args.sortOrder === -1
                                        ? "desc"
                                        : "sort"
                            }
                        />
                    );
                }}
                lazy
                {...props}
            />

        </MaximizeArea>
    </div>)
}