type HttpMethod =
  | "GET"
  | "POST"
  | "PUT"
  | "DELETE"
  | "PATCH"
  | "HEAD"
  | "OPTIONS";

/**
 * Perform a standard fetch operation with a _JSON_ object expected as a response.
 *
 * ## Example
 *
 * ```ts
 * // Set an interface for the expected response
 * interface RedirectEstate {
 *   redirect: string;
 * }
 *
 * // Handle response in 1 line
 * goFetch(url, postData, "POST")
 *     .then((data) => (window.location.href = decodeURIComponent((data as RedirectEstate).redirect)))
 *     .catch((e) => console.error(e));
 *
 * // OR Handle reponse within function body
 * goFetch(url, postData, "POST")
 *     .then((response) => {
 *         const data = response as RedirectEstate;
 *         window.location.href = decodeURIComponent(data.redirect);
 *     }
 *     .catch((e) => console.error(e));
 * ```
 */

export async function goFetch(
  url: string,
  body: object = {},
  method: HttpMethod = "GET",
  errorMessage: string | null = null,
  throwFullResponse: boolean = false,
): Promise<object> {
  const postBody = JSON.stringify(body);
  return fetch(url, {
    method: method,
    headers: {
      "Content-Type": "application/json",
      "x-from-inertia": "1",
    },
    body: method !== "GET" && method !== "HEAD" ? postBody : undefined,
  }).then(async (response) => {
    if (!response.ok) {
      console.warn(response);
      // console.warn(await response.text());
      console.table({
        url: response.url,
        status: response.status,
        message: response.statusText,
      });
      if (errorMessage !== null) {
        throw errorMessage;
      } else {
        if (throwFullResponse) {
          // console.log("Throwing full response...");
          throw await response.json();
        }
        throw `${response.statusText} with error code ${response.status}`;
      }
    }
    return response.json();
  });
}

/**
 * Shortcut to send a GET fetch request.
 * @param url Url enpoint to send the HTTP request to.
 * @param customErrorMessage A custom error message to throw as desired.
 * @returns `Promise<object>` Promise of the return object.
 */
export async function goGet(
  url: string,
  customErrorMessage: string | null = null,
): Promise<object> {
  return goFetch(url, {}, "GET", customErrorMessage);
}

/**
 * Shortcut to send a POST fetch request.
 * @param url Url enpoint to send the HTTP request to.
 * @param body The JSON object to send as params to the endpoint.
 * @param customErrorMessage A custom error message to throw as desired.
 * @returns `Promise<object>` Promise of the return object.
 */
export async function goPost(
  url: string,
  body: object = {},
  customErrorMessage: string | null = null,
  throwFullResponse: boolean = false,
): Promise<object> {
  return goFetch(url, body, "POST", customErrorMessage, throwFullResponse);
}

/**
 * Shortcut to send a PUT fetch request.
 * @param url Url enpoint to send the HTTP request to.
 * @param body The JSON object to send as params to the endpoint.
 * @param customErrorMessage A custom error message to throw as desired.
 * @returns `Promise<object>` Promise of the return object.
 */
export async function goPut(
  url: string,
  body: object = {},
  customErrorMessage: string | null = null,
): Promise<object> {
  return goFetch(url, body, "PUT", customErrorMessage);
}

/**
 * Shortcut to send a DELETE fetch request.
 * @param url Url enpoint to send the HTTP request to.
 * @param body The JSON object to send as params to the endpoint.
 * @param customErrorMessage A custom error message to throw as desired.
 * @returns `Promise<object>` Promise of the return object.
 */
export async function goDelete(
  url: string,
  body: object = {},
  customErrorMessage: string | null = null,
): Promise<object> {
  return goFetch(url, body, "DELETE", customErrorMessage);
}

/**
 * Perform a standard fetch operation with _plain text_ expected as a response.
 */

export async function goFetchText(
  url: string,
  body: object = {},
  method: HttpMethod = "GET",
  errorMessage: string | null = null,
): Promise<string> {
  const postBody = JSON.stringify(body);
  return fetch(url, {
    method: method,
    headers: {
      "Content-Type": "application/json",
      "x-from-inertia": "1",
    },
    body: method !== "GET" && method !== "HEAD" ? postBody : undefined,
  }).then(async (response) => {
    if (!response.ok) {
      console.warn(response);
      // console.warn(await response.text());
      console.table({
        url: response.url,
        status: response.status,
        message: response.statusText,
      });
      if (errorMessage !== null) {
        throw errorMessage;
      } else {
        throw `${response.statusText} with error code ${response.status}`;
      }
    }
    return response.text();
  });
}

/**
 * Perform a standard fetch operation with _binary blob_ expected as a response. An image for example.
 */

export async function goFetchBlob(
  url: string,
  body: object = {},
  method: HttpMethod = "GET",
  errorMessage: string | null = null,
): Promise<Blob> {
  const postBody = JSON.stringify(body);
  return fetch(url, {
    method: method,
    headers: {
      "Content-Type": "application/json",
      "x-from-inertia": "1",
    },
    body: method !== "GET" && method !== "HEAD" ? postBody : undefined,
  }).then(async (response) => {
    if (!response.ok) {
      console.warn(response);
      // console.warn(await response.text());
      console.table({
        url: response.url,
        status: response.status,
        message: response.statusText,
      });
      if (errorMessage !== null) {
        throw errorMessage;
      } else {
        throw `${response.statusText} with error code ${response.status}`;
      }
    }
    return response.blob();
  });
}
