import {
  DataTable,
  DataTableExpandedRows,
  DataTableFilterMeta,
  DataTableFilterMetaData,
  DataTableRowData,
  DataTableRowEvent,
  DataTableRowExpansionTemplate,
  DataTableRowToggleEvent,
  DataTableValueArray,
  SortOrder,
} from "primereact/datatable";
import React, {
  useState,
  useEffect,
  useCallback,
  ReactNode,
  CSSProperties,
  useRef,
} from "react";
import {
  ColumnBodyOptions,
  ColumnFilterElementTemplateOptions,
  ColumnProps,
} from "primereact/column";
import { InputNumber, InputNumberChangeEvent } from "primereact/inputnumber";
import {
  Dropdown,
  DropdownChangeEvent,
  DropdownProps,
} from "primereact/dropdown";
import { InputText } from "primereact/inputtext";
import { FilterMatchMode } from "primereact/api";
import { Skeleton } from "primereact/skeleton";
import { goGet } from "../../utils/goFetch";
import { Column } from "primereact/column";
import OSLabel from "../OSLabel/OSLabel";
import FAIcon from "../FAIcon";
import { FilterType } from "./FilterType";
import { Calendar } from "primereact/calendar";
import "./css/styles.css";

interface DataItem {
  id: number;
  // eslint-disable-next-line @typescript-eslint/no-explicit-any
  [key: string]: any;
}

interface Yii2DataProviderResponse {
  models: DataItem[];
  totalCount: number;
  page: number;
  pageSize: number;
}

export interface FilterDropDownOptions {
  value: string | number;
  label: string;
}

export interface TSColumnTemplate extends ColumnProps {
  field?: string;
  header?: string | React.ReactNode;
  sortable?: boolean;
  filter?: boolean;
  filterType?: FilterType;
  filterDropDownOptions?: FilterDropDownOptions[];
  dropdownFilter?: boolean;
  dropDownStatic?: boolean;
  dropDownValueTemplate?:
    | React.ReactNode
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    | ((option: any, props: DropdownProps) => React.ReactNode)
    | undefined;
  dropDownItemTemplate?:
    | React.ReactNode
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    | ((option: any) => React.ReactNode)
    | undefined;
  // eslint-disable-next-line @typescript-eslint/no-explicit-any
  body?: (rowData: any) => ReactNode;
  style?: CSSProperties;
  /**
   * Displays an icon to toggle row expansion.
   * @defaultValue false
   */
  // eslint-disable-next-line @typescript-eslint/no-explicit-any
  expander?: boolean | ((data: any, options: ColumnBodyOptions) => boolean);
}

interface Yii2DataTableProps {
  endpoint: string;
  columns: TSColumnTemplate[];
  model: string;
  primaryKey?: string | number;
  pagination: boolean;
  rowCount?: number;
  noHeader?: boolean;
  removableSort?: boolean | undefined;
  sortMode?: "single" | "multiple" | undefined;
  rowsPerPageOptions?: number[] | undefined;
  /**
   * Need to force a refresh of the table data without any other value changing? Simply increase this number.
   */
  refreshTrigger?: number;

  /**
   * A collection of rows or a map object row data keys that are expanded.
   */
  expandedRows?: DataTableValueArray | DataTableExpandedRows | undefined;
  /**
   * Callback to invoke when a row is toggled or collapsed.
   * @param {DataTableRowToggleEvent} event - Custom row toggle event.
   */
  onRowToggle?(event: DataTableRowToggleEvent): void;
  /**
   * Callback to invoke when a row is expanded.
   * @param {DataTableRowEvent} event - Custom row event.
   */
  onRowExpand?(event: DataTableRowEvent): void;
  /**
   * Callback to invoke when a row is collapsed.
   * @param {DataTableRowEvent} event - Custom row event.
   */
  onRowCollapse?(event: DataTableRowEvent): void;
  /**
   * Function that receives the row data as the parameter and returns the expanded row content. You can override the rendering of the content by setting options.customRendering = true.
   * @param {DataTableRowData<TValue>} data - Editing row data.
   * @param {DataTableRowExpansionTemplate} options - Options for the row expansion template.
   */
  rowExpansionTemplate?(
    // @ts-expect-error TValue not defined
    data: DataTableRowData<TValue>,
    options: DataTableRowExpansionTemplate
  ): React.ReactNode;
}

const OSDataTable: React.FC<Yii2DataTableProps> = ({
  endpoint,
  columns,
  model,
  primaryKey,
  pagination,
  rowCount = 20,
  noHeader,
  removableSort = undefined,
  sortMode = "single",
  rowsPerPageOptions = undefined,
  refreshTrigger = 0,
  expandedRows = undefined,
  onRowToggle = undefined,
  onRowExpand = undefined,
  onRowCollapse = undefined,
  rowExpansionTemplate = undefined,
}) => {
  /**
   *  Flag to determine whether current data is placeholder data or real data - Should never display placeholder data
   *  Standard variable as stays false after first render of component
   */
  const isDummyData = useRef(false);

  // Populates table with placeholder data to allow skeleton to display on initial load
  function generateSkeletonData(): DataItem[] {
    const data: DataItem[] = [];
    for (let index = 0; index < rowCount; index++) {
      data.push({ id: index });
    }
    isDummyData.current = true;
    return data;
  }

  const [data, setData] = useState<DataItem[]>(generateSkeletonData);
  const [loading, setLoading] = useState<boolean>(false);
  const [totalRecords, setTotalRecords] = useState<number>(0);
  const [first, setFirst] = useState<number>(0);
  const [rows, setRows] = useState<number>(rowCount);
  const [sortField, setSortField] = useState<string | undefined>(undefined);
  const [sortOrder, setSortOrder] = useState<SortOrder>(1);

  const [filters, setFilters] = useState<DataTableFilterMeta>(() => {
    const initialFilters: DataTableFilterMeta = {};

    columns.forEach((column) => {
      if (column.filter && column.field) {
        initialFilters[column.field] = {
          value: null,
          matchMode:
            column.filterType === FilterType.Number //Todo: Improve filter functionality
              ? FilterMatchMode.EQUALS
              : FilterMatchMode.CONTAINS,
        };
      }
    });

    return initialFilters;
  });

  const fetchData = useCallback(async () => {
    setLoading(true);
    try {
      const params = new URLSearchParams();

      if (pagination) {
        params.append("page", String(Math.floor(first / rows) + 1));
        params.append("pageSize", String(rows));
      }

      if (sortField && sortOrder) {
        params.append("sort", `${sortOrder === -1 ? "-" : ""}${sortField}`);
      }

      for (const [field, filter] of Object.entries(filters)) {
        const filterValue = (filter as DataTableFilterMetaData).value;
        if (filterValue !== undefined && filterValue !== null) {
          // recreate GET params as Yii would expect
          params.append(`${model}[${field}]`, filterValue.toString());
          // } else {
          //   console.log("Ignoring filtervalue...");
        }
      }

      //   console.log("Making request with params:", params.toString());

      const sendEndpoint = endpoint.includes("?")
        ? `${endpoint}&${params.toString()}`
        : `${endpoint}?${params.toString()}`;

      await goGet(sendEndpoint)
        .then((response) => {
          const responseData = response as {
            dataProvider: Yii2DataProviderResponse;
          };
          setData(responseData.dataProvider.models);
          setTotalRecords(responseData.dataProvider.totalCount);
        })
        .catch(() => {
          if (isDummyData.current) {
            setData([]);
          }
        })
        .finally(() => (isDummyData.current = false)); // Not necessary, but calls anyway to be safe
    } catch (error) {
      console.error("Error fetching data:", error);
    } finally {
      setLoading(false);
    }
  }, [endpoint, first, rows, sortField, sortOrder, filters, model, pagination]);

  useEffect(() => {
    fetchData();
  }, [fetchData, refreshTrigger]);

  const onPage = (event: { first: number; rows: number }) => {
    setFirst(event.first);
    setRows(event.rows);
  };

  const onSort = (event: { sortField: string; sortOrder: SortOrder }) => {
    setSortField(event.sortField);
    setSortOrder(event.sortOrder);
  };

  const onFilter = (event: { filters: DataTableFilterMeta }) => {
    // console.log("Filter event:", event.filters);
    setFilters(event.filters);
    setFirst(0);
  };

  const getFilterTemplate = (
    options: ColumnFilterElementTemplateOptions,
    column: TSColumnTemplate,
    filterDropDownOptions?: FilterDropDownOptions[]
  ) => {
    switch (column.filterType) {
      case FilterType.Number: {
        return (
          <InputNumber
            value={options.value || null}
            onChange={(e: InputNumberChangeEvent) => {
              options.filterApplyCallback(e.value);
            }}
            placeholder={`Filter by ${column.header}`}
            className="p-column-filter w-full"
            showButtons
            style={{ fontFamily: "Poppins" }}
          />
        );
      }

      case FilterType.Text: {
        return (
          <InputText
            value={options.value || ""}
            onChange={(e) => options.filterApplyCallback(e.target.value)}
            placeholder={`Filter by ${column.header}`}
            className="p-column-filter w-full"
            style={{ fontFamily: "Poppins" }}
          />
        );
      }

      case FilterType.Dropdown: {
        return (
          <Dropdown
            value={options.value || null}
            onChange={(e: DropdownChangeEvent) => {
              options.filterApplyCallback(e.value);
            }}
            options={filterDropDownOptions}
            optionLabel="label"
            placeholder={`Filter by ${column.header}`}
            className="p-column-filter"
            checkmark
            editable={
              column.dropDownStatic === true ||
              column.dropDownValueTemplate !== undefined
                ? false
                : true
            }
            filter={column.dropdownFilter}
            valueTemplate={column.dropDownValueTemplate}
            itemTemplate={column.dropDownItemTemplate}
          />
        );
      }

      case FilterType.Date: {
        return (
          <Calendar
            value={options.value || null}
            onChange={(e) => {
              options.filterApplyCallback(e.value);
            }}
            placeholder={`Filter by ${column.header}`}
            className="p-column-filter w-full"
            style={{ fontFamily: "Poppins" }}
            dateFormat="yy/mm/dd"
            readOnlyInput
            showButtonBar
          />
        );
      }
    }
  };

  const getHeaderTemplate = () => {
    let end = (Math.floor(first / rows) + 1) * rows;
    if (end > totalRecords) {
      end = totalRecords;
    }

    return (
      <>
        {data ? (
          <OSLabel
            props={{
              label:
                end !== 0
                  ? pagination
                    ? `Showing ${
                        first + 1
                      } to ${end} out of ${totalRecords} total records`
                    : `${totalRecords} records found`
                  : "No records found",
              labelWeight: 500,
              labelFontSize: "1.6rem",
            }}
          />
        ) : (
          <Skeleton height="10rem" />
        )}
      </>
    );
  };

  return (
    <div id="table-wrapper" className="flex p-3 w-full">
      <DataTable
        id="os-data-table"
        value={data}
        lazy
        paginator={pagination && totalRecords > rows}
        rows={pagination ? rows : undefined}
        rowsPerPageOptions={rowsPerPageOptions}
        totalRecords={totalRecords}
        first={first}
        onPage={onPage}
        onSort={onSort}
        sortField={sortField}
        sortOrder={sortOrder}
        filters={filters}
        filterDisplay="row"
        filterClearIcon={() => {
          return (
            <div className="mt-1">
              <FAIcon
                identifier="filter-circle-xmark"
                style={{ margin: "0rem", color: "var(--danger-red)" }}
                size={"lg"}
              />
            </div>
          );
        }}
        onFilter={onFilter}
        key={primaryKey}
        emptyMessage="No records found"
        stripedRows
        header={noHeader ? undefined : getHeaderTemplate}
        className="w-full"
        removableSort={removableSort}
        sortMode={sortMode}
        expandedRows={expandedRows}
        onRowToggle={onRowToggle}
        onRowExpand={onRowExpand}
        onRowCollapse={onRowCollapse}
        rowExpansionTemplate={rowExpansionTemplate}
      >
        {columns.map((column, i) => (
          <Column
            key={i}
            field={column.field}
            header={column.header || <></>}
            sortable={column.sortable}
            showFilterMenu={false}
            filter={column.filter}
            filterField={column.field}
            filterElement={(options) =>
              getFilterTemplate(options, column, column.filterDropDownOptions)
            }
            body={
              loading || isDummyData.current ? (
                <>
                  {" "}
                  <Skeleton height="3rem" />{" "}
                </>
              ) : column.body ? (
                (rowData) => column.body!(rowData)
              ) : undefined
            }
            style={column.style}
            expander={column.expander}
          />
        ))}
      </DataTable>
    </div>
  );
};

export default OSDataTable;
