import React from "react";
import { Button } from "primereact/button";
import FAIcon from "../FAIcon";
import { FAIdentifier } from "../../types/fa-icons";
import { Divider } from "primereact/divider";
import { ReactNode } from "react";

export interface OSActionButtonProps {
    /**
     * Predefined style types
     */
    type: OSActionButtonType;

    label: string;
    onClick: React.MouseEventHandler;

    /**
     * Whether the icon before the label is enabled.
     */
    primaryIconEnabled: boolean;
    primaryIcon?: FAIdentifier;
    secondaryIconEnabled?: boolean;
    secondaryIcon?: FAIdentifier;

    /**
     * Whether there should be a vertical line after the label and before the secondary icon.
     */
    secondaryIconDivider?: boolean;

    /**
     * Currently only affects the Filter button style.
     * Intended to switch between solid and outlined with a white background.
     */
    outlined?: boolean;
}

export const PrimaryButtonStyle = {
    minWidth: 'max-content',
    width: 'fit-content',
    borderRadius: '5px',
    border: '1px solid #1fa55e',
    backgroundColor: '#1fa55e',
    padding: '0rem',
};

export const SecondaryButtonStyle = {
    minWidth: 'max-content',
    border: '1px solid #1fa55e',
    borderRadius: '5px',
    backgroundColor: 'white',
    padding: '0rem'
};

export const TertiaryButtonStyle = {
    minWidth: 'max-content',
    border: '1px solid var(--danger-red)',
    borderRadius: '5px',
    backgroundColor: 'var(--danger-red)',
    padding: '0rem'
};

export const CancelButtonStyle = {
    minWidth: 'max-content',
    border: '1px solid var(--smokey-grey)',
    borderRadius: '5px',
    backgroundColor: 'var(--smokey-grey)',
    padding: '0rem'
};

export const FilterButtonStyle = {
    minWidth: 'max-content',
    border: '1px solid var(--main-colour)',
    borderRadius: '10px',
    backgroundColor: 'white',
    color: 'var(--main-colour)',
    padding: '0rem'
}

export enum OSActionButtonType {
    Primary = "primary",
    Secondary = "secondary",
    Tertiary = "tertiary",
    Cancel = "cancel",
    Filter = "filter"
};

const OSActionButton: React.FC<{ props: OSActionButtonProps, children?: ReactNode }> = ({ props, children }) => {

    const faceColour = () => {
        switch (props.type) {
            case OSActionButtonType.Primary: {
                return 'white';
            }
            case OSActionButtonType.Secondary: {
                return '#1fa55e';
            }
            case OSActionButtonType.Filter: {
                if (props.outlined) {
                    return 'var(--main-colour)';
                }
                return 'white';
            }
            default: {
                return 'white';
            }
        }
    }

    const buttonStyle = () => {
        switch (props.type) {
            case OSActionButtonType.Primary: {
                return PrimaryButtonStyle;
            };
            case OSActionButtonType.Secondary: {
                return SecondaryButtonStyle;
            };
            case OSActionButtonType.Tertiary: {
                return TertiaryButtonStyle;
            };
            case OSActionButtonType.Cancel: {
                return CancelButtonStyle;
            }
            case OSActionButtonType.Filter: {
                const filterStyle = FilterButtonStyle;
                if (props.outlined) {
                    ;
                    filterStyle.backgroundColor = 'white';
                    filterStyle.color = 'var(--main-colour)';
                } else {
                    filterStyle.backgroundColor = 'var(--main-colour)';
                    filterStyle.color = 'white'
                }
                return filterStyle;
            }
            default: {
                return PrimaryButtonStyle;
            }
        }
    }

    const iconStyle = {
        color: faceColour(),
        margin: '0rem',
        padding: '0rem',
        fontSize: '1.2rem',
        fontWeight: '500',
        lineHeight: 1
    };

    const textStyle = {
        color: faceColour(),
        margin: '0rem',
        padding: '0rem',
        fontFamily: 'Poppins',
        fontSize: '1.2rem',
        fontWeight: '500'
    };

    return (
        <>
            <Button style={buttonStyle()} unstyled={true} onClick={props.onClick}>
                {children ? children : <div className="flex flex-row gap-3 justify-center items-stretch" style={{ fontSize: '1.5rem' }}>
                    {props.primaryIconEnabled ? (<div className="flex items-center py-3 pl-3">
                        {props.primaryIcon ? (
                            <FAIcon identifier={props.primaryIcon} style={iconStyle} />
                        ) : null}
                    </div>) : null}
                    <div className={"flex items-center py-3 " + (!props.secondaryIconEnabled ? "pr-3" : "") + (!props.primaryIconEnabled ? "pl-3" : "")}>
                        <p style={textStyle}>{props.label}</p>
                    </div>
                    {props.secondaryIconEnabled ? (
                        <div className="flex items-center pr-3">
                            {props.secondaryIconDivider ? (
                                <Divider layout="vertical" className="mr-3 ml-0 mt-0 mb-0" />
                            ) : null}
                            {props.secondaryIcon ? (
                                <FAIcon identifier={props.secondaryIcon} style={iconStyle} />
                            ) : null}
                        </div>
                    ) : null}
                </div>}
            </Button>
        </>
    );
}

export default OSActionButton;