import React, { ChangeEventHandler, useState } from "react";
import { modelProps } from "../types/props";
import {
  Form,
  FormControlProps,
  FormGroup,
  FormSelectProps,
  InputGroup,
} from "react-bootstrap";
import {
  SetErrorValidation,
  setFormId,
  setGroupClass,
  setHints,
  setLabels,
  setValue,
} from "../helpers/InputHelpers";

interface TextInputProps extends FormControlProps {
  model: modelProps;
  field: string;
  type?: string;
  label?: string | boolean | null;
  hint?: string | boolean | null;
  groupClass?: string | null;
  onChange?: ChangeEventHandler;
  errorMessage?: string | null;
  inputGroupPre?: string | false;
  inputGroupPost?: string | false;
}

/******* TEXT INPUT *******/

export const TextInput: React.FC<TextInputProps> = ({
  type = "text",
  label,
  model,
  field,
  groupClass,
  hint = false,
  errorMessage = null,
  onChange,
  inputGroupPre = false,
  inputGroupPost = false,
  ...props
}) => {
  label = setLabels(label, field, model);
  hint = setHints(hint, field, model);
  groupClass = setGroupClass(groupClass);
  const value = setValue(field, model);
  const formId = setFormId(model.name, field);

  const rules = model.rules as unknown[];
  let required = false;
  rules.forEach((rule) => {
    // @ts-expect-error rule[0] not TS defined.
    const fields = rule[0] as string[];
    if (!fields.includes(field)) {
      return;
    }

    // @ts-expect-error rule[1] not TS defined.
    if (rule[1]) {
      // @ts-expect-error rule[1] not TS defined.
      if (rule[1] === "required") {
        required = true;
      }
    }
  });

  const [errorClass, setErrorClass] = useState("");
  const [formValidation, setFormValidation] = useState("");
  SetErrorValidation(setErrorClass, setFormValidation, errorMessage);

  const useInputGroup =
    (inputGroupPre !== false && inputGroupPre !== undefined) ||
    (inputGroupPost !== false && inputGroupPost !== undefined);

  return (
    <>
      <FormGroup className={groupClass} controlId={formId}>
        {label !== false ? <Form.Label>{label}</Form.Label> : null}

        {useInputGroup ? (
          <InputGroup>
            {inputGroupPre ? (
              <InputGroup.Text>{inputGroupPre}</InputGroup.Text>
            ) : null}

            <Form.Control
              type={type}
              required={required}
              defaultValue={value}
              onChange={onChange}
              className={formValidation}
              {...props}
            />

            {inputGroupPost ? (
              <InputGroup.Text>{inputGroupPost}</InputGroup.Text>
            ) : null}
          </InputGroup>
        ) : (
          <Form.Control
            type={type}
            required={required}
            defaultValue={value}
            onChange={onChange}
            className={formValidation}
            {...props}
          />
        )}

        {hint !== false ? (
          <Form.Text className="text-muted">{hint}</Form.Text>
        ) : null}
        <Form.Text id={`${formId}-validation`} className={errorClass}>
          {errorMessage}
        </Form.Text>
      </FormGroup>
    </>
  );
};

/******* SELECT INPUT *******/

export interface SelectOptionsProp {
  [key: string | number]: string;
}

interface SelectInputProps extends FormSelectProps {
  model: modelProps;
  field: string;
  options: SelectOptionsProp;
  label?: string | boolean | null;
  hint?: string | boolean | null;
  groupClass?: string | null;
  onChange?: ChangeEventHandler;
  errorMessage?: string | null;
}

export const SelectInput: React.FC<SelectInputProps> = ({
  label,
  model,
  field,
  options,
  groupClass,
  hint = false,
  errorMessage = null,
  onChange,
  ...props
}) => {
  label = setLabels(label, field, model);
  hint = setHints(hint, field, model);
  groupClass = setGroupClass(groupClass);
  const value = setValue(field, model);
  const formId = setFormId(model.name, field);

  const rules = model.rules as unknown[];
  let required = false;
  rules.forEach((rule) => {
    // @ts-expect-error rule[0] not TS defined.
    const fields = rule[0] as string[];
    if (!fields.includes(field)) {
      return;
    }

    // @ts-expect-error rule[1] not TS defined.
    if (rule[1]) {
      // @ts-expect-error rule[1] not TS defined.
      if (rule[1] === "required") {
        required = true;
      }
    }
  });

  const [errorClass, setErrorClass] = useState("");
  const [formValidation, setFormValidation] = useState("");
  SetErrorValidation(setErrorClass, setFormValidation, errorMessage);

  return (
    <>
      <FormGroup className={groupClass} controlId={formId}>
        {label !== false ? <Form.Label>{label}</Form.Label> : null}
        <Form.Select
          required={required}
          onChange={onChange}
          className={formValidation}
          defaultValue={value}
          {...props}
        >
          <option value="">Select</option>
          {Object.keys(options).map((key) => (
            <option value={key} key={key}>
              {options[key]}
            </option>
          ))}
        </Form.Select>
        {hint !== false ? (
          <Form.Text className="text-muted">{hint}</Form.Text>
        ) : null}
        <Form.Text id={`${formId}-validation`} className={errorClass}>
          {errorMessage}
        </Form.Text>
      </FormGroup>
    </>
  );
};

/******* HIDDEN INPUT *******/

interface HiddenInputProps {
  model: modelProps;
  field: string;
  onChange?: ChangeEventHandler;
}

export const HiddenInput: React.FC<HiddenInputProps> = ({
  field,
  model,
  onChange,
}) => (
  <>
    <FormGroup controlId={setFormId(model.name, field)}>
      <Form.Control
        type="hidden"
        defaultValue={setValue(field, model)}
        onChange={onChange}
      />
    </FormGroup>
  </>
);
