<?php

namespace onespace\tools\rabbit\services\base;

use Interop\Amqp\Impl\AmqpMessage;
use onespace\tools\rabbit\services\interfaces\ServiceInterface;
use stdClass;

abstract class BaseRabbitService implements ServiceInterface {
    protected readonly AmqpMessage $message;
    protected readonly string $rawBody;
    protected readonly array $headers;
    protected readonly ?string $routingKey;

    /**
     * Where the decoded $rawBody is contained.
     *
     * @readonly
     * @access  protected
     */

    protected readonly array|stdClass $body;

    /**
     * Holds any error messages recorded by this service
     *
     * @var array   $errorMessages  Default: []
     *
     * @access  protected
     */

    protected array $errorMessages = [];

    /**
     * Each class must declare a class to decode the body of the message.
     *
     * This cannot be done directly here because either JSON or Protobufs
     * may be sent down.
     *
     * @return  array|stdClass
     *
     * @access  public
     */
    abstract public function decodeRawBody(): array|stdClass;

    /**
     * {@inheritdoc}
     */
    public function __construct(AmqpMessage $message) {
        $this->message = $message;
        $this->rawBody = $message->getBody();
        $this->headers = $message->getHeaders();
        $this->routingKey = $message->getRoutingKey();

        $this->body = $this->decodeRawBody();
    }

    /**
     * Get all recorded error messages.
     *
     * @return  array
     *
     * @access  public
     */
    public function getErrorMessages(): array {
        return $this->errorMessages;
    }

    /**
     * Sets an error message.
     *
     * @param   mixed   $message
     * @param   string|int|null $id Default is null
     *
     * @access  public
     */
    public function addErrorMessage(mixed $message, string|int|null $id = null): void {
        if ($id === null) {
            $this->errorMessages[] = $message;
        } else {
            $this->errorMessages[$id][] = $message;
        }
    }

    /**
     * Returns the `headers` property.
     *
     * @return  array
     *
     * @access  public
     */
    public function getHeaders(): array {
        return $this->headers;
    }

    /**
     * Returns the `routingKey` property.
     *
     * @return string|null
     *
     * @access  public
     */
    public function getRoutingKey(): ?string {
        return $this->routingKey;
    }

    /**
     * Returns the `body` property.
     *
     * @return  array|stdClass
     *
     * @access  public
     */
    public function getBody(): array|stdClass {
        return $this->body;
    }
}
