<?php

namespace onespace\tools\rabbit\helpers;

trait CLIHelper {
    /**
     * Write to the screen a parsed string _WITHOUT_ a trailing line break.
     */
    protected function write(string $string): void {
        print_r($string);
    }

    /**
     * Write to the screen a parsed string _WITH_ a trailing line break.
     */
    protected function writeln(string $string): void {
        print_r($string . PHP_EOL);
    }

    /**
     * Write to the screen an object or array of data, formatted in a readable way.
     */
    protected function writeData(array|object $data): void {
        print_r($data);
    }

    /**
     * Write "true" or "false" to the screen based on a parsed bool value _WITHOUT_ a trailing line break.
     */
    protected function writeBool(bool $bool): void {
        if ($bool) {
            $this->write('true');
        } else {
            $this->write('false');
        }
    }

    /**
     * Write "true" or "false" to the screen based on a parsed bool value _WITH_ a trailing line break.
     */
    protected function writelnBool(bool $bool): void {
        if ($bool) {
            $this->writeln('true');
        } else {
            $this->writeln('false');
        }
    }

    /**
     * Write a "✔" or "✘" to the screen based on a parsed bool value _WITHOUT_ a trailing line break.
     */
    protected function writeCheck(bool $bool): void {
        if ($bool) {
            $this->write('✔');
        } else {
            $this->write('✘');
        }
    }

    /**
     * Write a "✔" or "✘" to the screen based on a parsed bool value _WITH_ a trailing line break.
     */
    protected function writelnCheck(bool $bool): void {
        if ($bool) {
            $this->writeln('✔');
        } else {
            $this->writeln('✘');
        }
    }

    /**
     * Write a date & time Timestamp _WITHOUT_ a trailing line break.
     */
    protected function writeTs(string $format = 'Y-m-d H:i:s'): void {
        $this->write(date($format));
    }

    /**
     * Write a date & time Timestamp _WITH_ a trailing line break.
     */
    protected function writelnTs(string $format = 'Y-m-d H:i:s'): void {
        $this->writeln(date($format));
    }

    /**
     * Draw a series of `.` to make a line.
     *
     * @param   int $count  The number of characters in the line.
     */
    protected function drawDotLine(int $count = 22): void {
        $this->writeln($this->repeatChar('.', $count));
    }

    /**
     * Draw a series of `-` to make a line.
     *
     * @param   int $count  The number of characters in the line.
     */
    protected function drawDashLine(int $count = 22): void {
        $this->writeln($this->repeatChar('-', $count));
    }

    /**
     * Draw a series of `+` to make a line.
     *
     * @param   int $count  The number of characters in the line.
     */
    protected function drawPlusLine(int $count = 22): void {
        $this->writeln($this->repeatChar('+', $count));
    }

    /**
     * Draw a series of `*` to make a line.
     *
     * @param   int $count  The number of characters in the line.
     */
    protected function drawStarLine(int $count = 22): void {
        $this->writeln($this->repeatChar('*', $count));
    }

    /**
     * Draw a series of `_` to make a line.
     *
     * @param   int $count  The number of characters in the line.
     */
    protected function drawUnderscoreLine(int $count = 22): void {
        $this->writeln($this->repeatChar('_', $count));
    }

    /**
     * Draw a series of `#` to make a line.
     *
     * @param   int $count  The number of characters in the line.
     */
    protected function drawHashLine(int $count = 22): void {
        $this->writeln($this->repeatChar('#', $count));
    }

    private function repeatChar(string $char, int $count): string {
        $str = '';
        for ($i = 0; $i < $count; $i++) {
            $str .= $char;
        }
        return $str;
    }

    private function setMaxLen(array &$maxLen, string $heading, string $value): void {
        $len = strlen($value);
        if (!isset($maxLen[$heading])) {
            $maxLen[$heading] = $len;
            return;
        }
        if ($len > $maxLen[$heading]) {
            $maxLen[$heading] = $len;
        }
    }

    /**
     * Draw out a table on the CLI of parsed headings and values.
     */
    protected function table(array $headings, array $values): void {
        $maxLen = [];
        foreach ($headings as $heading) {
            $maxLen[$heading] = strlen($heading);
        }
        foreach ($values as $key => $value) {
            if (is_array($value)) {
                foreach ($value as $k => $v) {
                    $this->setMaxLen($maxLen, $k, $v);
                }
            } else {
                $this->setMaxLen($maxLen, $key, $value);
            }
        }

        $str = '';

        foreach ($headings as $heading) {
            $str .= "| {$heading} ";
            if (strlen($heading) < $maxLen[$heading]) {
                $str .= $this->repeatChar(' ', $maxLen[$heading] - strlen($heading));
            }
        }
        $str .= "|\n";

        $str .= "|";
        foreach ($headings as $heading) {
            $str .= $this->repeatChar('-', $maxLen[$heading] + 2);
            $str .= '+';
        }
        $str = substr_replace($str, "|\n", -1);

        $multi = false;
        foreach ($values as $key => $value) {
            if (is_array($value)) {
                $multi = true;
                foreach ($value as $k => $v) {
                    $str .= "| {$v} ";
                    $str .= $this->repeatChar(' ', $maxLen[$k] - strlen($v));
                }
                $str .= "|\n";
            } else {
                $str .= "| {$value} ";
                $str .= $this->repeatChar(' ', $maxLen[$key] - strlen($value));
            }
        }
        if (!$multi) {
            $str .= "|\n";
        }
        $this->writeln($str);
    }

    /**
     * In debug mode, print the data to the screen.
     */
    protected function printProtoFile(\Google\Protobuf\Internal\Message $bin): void {
        $this->writeln($bin->serializeToJsonString() . PHP_EOL);
        $this->writeln(bin2hex($bin->serializeToString()) . PHP_EOL);
    }
}
