<?php

namespace onespace\tools\rabbit\helpers;


use app\services\rabbit\RabbitService;
use Enqueue\AmqpLib\AmqpConsumer;
use Exception;
use Interop\Amqp\Impl\AmqpMessage;
use onespace\tools\rabbit\components\Rabbit;
use onespace\tools\rabbit\models\RabbitOutbox;
use Yii;

/**
 * Generalized common methods to be used in a standard `command\RabbitController.php`
 * 
 * @author  Gareth Palmer
 */
trait RabbitControllerHelper {

    /**
     * @note - Until PHP 8.2, the constants will need to be defined in each child class. In the meantime
     * it is here ready for 8.2 use.
     */
    // public const APP_PREFIX = 'v1_yourservice_';
    // public const DEFAULT_QUEUE_COUNT = 1;
    // public const AVERAGE_SECS_PER_MESSAGE = 0.1;

    /**
     * The path of your rabbit service classes. Overwrite in the parent class if a different path is
     * required - for example in yii2-advanced split.
     * 
     * @var string  $servicePath    Default value: `\app\services\rabbit\`
     * 
     * @access  protected
     */
    protected string $servicePath = '\app\services\rabbit\\';


    /**
     * Get a definitive standard queue name from a parsed queue name.
     * 
     * @param   string  $name   The raw queue name to be used.
     * 
     * @return  string
     * 
     * @static
     * @access  protected
     */

    protected static function getQueueName(string $name): string {
        $prefix = '';
        $name = str_replace('-', '_', $name);
        if (YII_DEBUG) {
            // $prefix = 'dev_';
        }
        return $prefix . static::APP_PREFIX . $name;
    }


    /**
     * Get a definitive standard exchange name from a parsed echange name.
     * 
     * @param   string  $name   The raw echanged name to be used.
     * 
     * @return  string
     * 
     * @static
     * @access  protected
     */

    protected static function getExchangeName(string $name): string {
        $prefix = '';
        $name = str_replace('-', '_', $name);
        if (YII_DEBUG) {
            // $prefix = 'dev_';
        }
        return $prefix . static::APP_PREFIX . $name;
    }


    /**
     * Returns the queue count to spin up on a typical service. Either from a parsed queue name in
     * the .env or the default defined in the parent class.
     * 
     * @param   string  $name   The identifier used to get the .env variable. Usually the entity name.
     * 
     * @return  int
     * 
     * @static
     * @access  protected
     */

    protected static function getQueueCount(string $name): int {
        $name = strtoupper($name);
        if (($count = getenv($name . '_QUEUE_COUNT')) !== false) {
            return $count;
        }
        return static::DEFAULT_QUEUE_COUNT;
    }


    /**
     * Monitor the Rabbit outbox, if anything is there, send them.
     */

    public function actionMonitorOutbox() {
        echo "Monitoring outbox...\n";
        echo "Connecting to Rabbit server...\n";
        /** @var Rabbit $rabbit */
        $rabbit = Yii::$app->rabbit;
        $rabbit->connect();
        echo "Connected. Monitoring...\n";
        while (true) {
            // check if there is anything in the outbox table
            $messages = RabbitOutbox::find();
            if ($messages->count() > 0) {
                // if so, send them on Rabbit
                foreach ($messages->each() as $message) {
                    /** @var RabbitOutbox $message */
                    try {
                        $rabbit->sendMessage(
                            destination: $rabbit->declareExchange(
                                $message->exchange,
                                Rabbit::EXCHANGE_TYPE_TOPIC,
                                [Rabbit::EXCHANGE_FLAG_DURABLE, Rabbit::EXCHANGE_FLAG_PASSIVE]
                            ),
                            message: $message->message,
                            routingKey: $message->routingKey,
                            customHeaders: $message->headers,
                            timestamp: $message->timestamp,
                            contentType: $message->contentType,
                            persistent: ($message->persistent ? true : false),
                            priority: $message->priority,
                            ttl: $message->ttl
                        );

                        $message->delete();
                        echo "Sent message #{$message->id}!\n";
                    } catch (Exception $ex) {
                        echo "Failed to send message #{$message->id}!\n";
                        echo "\t{$ex->getMessage()}\n";
                        // wait a second to prevent overloading the server
                        sleep(1);
                        // break at this point - need to maintain FIFO
                        break;
                    }
                }
            } else {
                // if there is nothing, wait a second before trying again
                sleep(1);
            }
        }
    }


    /**
     * Listen for entity events on the queue, and process them as they come in
     * 
     * @param   string  $entityName The identifier of the EventService to to execute. The file should be `Process{$entityName}EventService.php`.
     * @param   int     $timeout    When the run should time out. Typically passed as 0 for unlimited time.
     * 
     * @access  public
     */

    public function actionListenEntityEvents(string $entityName, int $timeout): void {
        echo "Starting...\n";
        /** @var Rabbit $rabbit */
        $rabbit = Yii::$app->rabbit;

        $className = $this->servicePath . 'Process' . str_replace(' ', '', ucwords(str_replace(['-', '.'], ' ', $entityName))) . 'EventService';
        if (!class_exists($className)) {
            echo "Failed to find class $className";
            return false;
        }

        $queues = [];
        $baseQueueName = self::getQueueName($entityName) . '_';
        // prep the names of all the queues to connect to
        echo "Connecting to queues...\n";
        for ($x = 1; $x <= self::getQueueCount($entityName); $x++) {
            // create a 4 digit number out of the index
            $stringNum = str_pad(strval($x), 4, '0', STR_PAD_LEFT);

            // try to load the queue if it already exists
            // if the queue doesn't exist for any reason
            if ($queue = $rabbit->loadExistingQueue($baseQueueName . $stringNum)) {
                echo "\tFound queue {$baseQueueName}{$stringNum}...\n";
                $queues[] = $queue;
            }
        }
        // create a consumer that will listen to all the queues
        $consumer = $rabbit->subscribeToMessages($queues, function (AmqpMessage $message, AmqpConsumer $consumer) use ($className) {
            /** @var RabbitService $service */
            $service = new $className($message->getBody(), $message->getRoutingKey());

            $result = $service->run();

            // depending on the result, either acknowledge or reject the message
            if ($result) {
                echo "Acknowledged!\n";
                $consumer->acknowledge($message);
            } else {
                echo implode('; ', $service->getErrorMessages());
                $consumer->reject($message, true);
            }

            return true;
        });
        echo "Listening to queues...\n";
        // start listening
        while (true) {
            $consumer->consume($timeout);
        }
    }


    /**
     * Listen for user events on the queue, and process them as they come in
     * 
     * @param   int     $timeout    When the run should time out. Typically passed as 0 for unlimited time.
     * @param   string  $queueNames A comma seperated string with a list of queues to monitor.
     * @param   string  $methodName The method to call when one of the queues that are monitored receives a message.
     *                              This method should be in the parent class.
     * 
     * @access  public
     */

    public function actionListenArbitraryEvents(int $timeout, string $queueNames, string $methodName): void {
        echo "Starting...\n";
        /** @var Rabbit $rabbit */
        $rabbit = Yii::$app->rabbit;

        // prep the names of all the queues to connect to
        echo "Connecting to queues...\n";
        $queues = [];
        foreach (explode(',', $queueNames) as $queueName) {
            // try to load the queue if it already exists
            // if the queue doesn't exist for any reason
            if ($queue = $rabbit->loadExistingQueue($queueName)) {
                echo "\tFound queue $queueName...\n";
                $queues[] = $queue;
            }
        }
        // create a consumer that will listen to all the queues
        $consumer = $rabbit->subscribeToMessages($queues, function (AmqpMessage $message, AmqpConsumer $consumer) use ($methodName) {
            return $this->{$methodName}($message, $consumer);
        });
        echo "Listening to queues...\n";
        // start listening
        while (true) {
            $consumer->consume($timeout);
        }
    }
}
