<?php

namespace onespace\tools\rabbit\helpers;

trait CLIHelper {
    /**
     * Write to the screen a parsed string _WITHOUT_ a trailing line break.
     */
    protected function write(string $string, Log $log = Log::NONE): void {
        if ($log === Log::DEBUG) {
            if (YII_DEBUG) {
                print_r($log->level() . $string);
            }
        } else {
            print_r($log->level() . $string);
        }
    }

    /**
     * Write to the screen a parsed string _WITH_ a trailing line break.
     */
    protected function writeln(string $string, Log $log = Log::NONE): void {
        if ($log === Log::DEBUG) {
            if (YII_DEBUG) {
                print_r($log->level() . $string . PHP_EOL);
            }
        } else {
            print_r($log->level() . $string . PHP_EOL);
        }
    }

    /**
     * Write to the screen an object or array of data, formatted in a readable way.
     */
    protected function writeData(array|object $data, Log $log = Log::NONE): void {
        if ($log === Log::DEBUG) {
            if (YII_DEBUG) {
                $this->writeln($log->level());
                print_r($data);
            }
        } else {
            $this->writeln($log->level());
            print_r($data);
        }
    }

    /**
     * Write "true" or "false" to the screen based on a parsed bool value _WITHOUT_ a trailing line break.
     */
    protected function writeBool(bool $bool, Log $log = Log::NONE): void {
        if ($bool) {
            $this->write('true', $log);
        } else {
            $this->write('false', $log);
        }
    }

    /**
     * Write "true" or "false" to the screen based on a parsed bool value _WITH_ a trailing line break.
     */
    protected function writelnBool(bool $bool, Log $log = Log::NONE): void {
        if ($bool) {
            $this->writeln('true', $log);
        } else {
            $this->writeln('false', $log);
        }
    }

    /**
     * Write a "✔" or "✘" to the screen based on a parsed bool value _WITHOUT_ a trailing line break.
     */
    protected function writeCheck(bool $bool, Log $log = Log::NONE): void {
        if ($bool) {
            $this->write('✔', $log);
        } else {
            $this->write('✘', $log);
        }
    }

    /**
     * Write a "✔" or "✘" to the screen based on a parsed bool value _WITH_ a trailing line break.
     */
    protected function writelnCheck(bool $bool, Log $log = Log::NONE): void {
        if ($bool) {
            $this->writeln('✔', $log);
        } else {
            $this->writeln('✘', $log);
        }
    }

    /**
     * Write a date & time Timestamp _WITHOUT_ a trailing line break.
     */
    protected function writeTs(string $format = 'Y-m-d H:i:s', Log $log = Log::NONE): void {
        $this->write(date($format), $log);
    }

    /**
     * Write a date & time Timestamp _WITH_ a trailing line break.
     */
    protected function writelnTs(string $format = 'Y-m-d H:i:s', Log $log = Log::NONE): void {
        $this->writeln(date($format), $log);
    }

    /**
     * Draw a series of `.` to make a line.
     *
     * @param   int $count  The number of characters in the line.
     * @param   string|null $wrapper    A wrapping character or characters to put at
     *                                  the beginning and end of the line, for example `|`.
     *                                  Set to null to skip. Default: `null`
     */
    protected function drawDotLine(int $count = 22, ?string $wrapper = null, Log $log = Log::NONE): void {
        $txt = "";
        if ($wrapper !== null) {
            $txt .= $wrapper;
        }
        $txt .= $this->repeatChar('.', $count);
        if ($wrapper !== null) {
            $txt .= $wrapper;
        }
        $this->writeln($txt, $log);
    }

    /**
     * Draw a series of `-` to make a line.
     *
     * @param   int $count  The number of characters in the line.
     * @param   string|null $wrapper    A wrapping character or characters to put at
     *                                  the beginning and end of the line, for example `|`.
     *                                  Set to null to skip. Default: `null`
     */
    protected function drawDashLine(int $count = 22, ?string $wrapper = null, Log $log = Log::NONE): void {
        $txt = "";
        if ($wrapper !== null) {
            $txt .= $wrapper;
        }
        $txt .= $this->repeatChar('-', $count);
        if ($wrapper !== null) {
            $txt .= $wrapper;
        }
        $this->writeln($txt, $log);
    }

    /**
     * Draw a series of `+` to make a line.
     *
     * @param   int $count  The number of characters in the line.
     * @param   string|null $wrapper    A wrapping character or characters to put at
     *                                  the beginning and end of the line, for example `|`.
     *                                  Set to null to skip. Default: `null`
     */
    protected function drawPlusLine(int $count = 22, ?string $wrapper = null, Log $log = Log::NONE): void {
        $txt = "";
        if ($wrapper !== null) {
            $txt .= $wrapper;
        }
        $txt .= $this->repeatChar('+', $count);
        if ($wrapper !== null) {
            $txt .= $wrapper;
        }
        $this->writeln($txt, $log);
    }

    /**
     * Draw a series of `*` to make a line.
     *
     * @param   int $count  The number of characters in the line.
     * @param   string|null $wrapper    A wrapping character or characters to put at
     *                                  the beginning and end of the line, for example `|`.
     *                                  Set to null to skip. Default: `null`
     */
    protected function drawStarLine(int $count = 22, ?string $wrapper = null, Log $log = Log::NONE): void {
        $txt = "";
        if ($wrapper !== null) {
            $txt .= $wrapper;
        }
        $txt .= $this->repeatChar('*', $count);
        if ($wrapper !== null) {
            $txt .= $wrapper;
        }
        $this->writeln($txt, $log);
    }

    /**
     * Draw a series of `_` to make a line.
     *
     * @param   int $count  The number of characters in the line.
     * @param   string|null $wrapper    A wrapping character or characters to put at
     *                                  the beginning and end of the line, for example `|`.
     *                                  Set to null to skip. Default: `null`
     */
    protected function drawUnderscoreLine(int $count = 22, ?string $wrapper = null, Log $log = Log::NONE): void {
        $txt = "";
        if ($wrapper !== null) {
            $txt .= $wrapper;
        }
        $txt .= $this->repeatChar('_', $count);
        if ($wrapper !== null) {
            $txt .= $wrapper;
        }
        $this->writeln($txt, $log);
    }

    /**
     * Draw a series of `#` to make a line.
     *
     * @param   int $count  The number of characters in the line.
     * @param   string|null $wrapper    A wrapping character or characters to put at
     *                                  the beginning and end of the line, for example `|`.
     *                                  Set to null to skip. Default: `null`
     */
    protected function drawHashLine(int $count = 22, ?string $wrapper = null, Log $log = Log::NONE): void {
        $txt = "";
        if ($wrapper !== null) {
            $txt .= $wrapper;
        }
        $txt .= $this->repeatChar('#', $count);
        if ($wrapper !== null) {
            $txt .= $wrapper;
        }
        $this->writeln($txt, $log);
    }

    private function repeatChar(string $char, int $count): string {
        $str = '';
        for ($i = 0; $i < $count; $i++) {
            $str .= $char;
        }
        return $str;
    }

    private function setMaxLen(array &$maxLen, string $heading, string $value): void {
        $len = strlen($value);
        if (!isset($maxLen[$heading])) {
            $maxLen[$heading] = $len;
            return;
        }
        if ($len > $maxLen[$heading]) {
            $maxLen[$heading] = $len;
        }
    }

    /**
     * Draw out a table on the CLI of parsed headings and values.
     */
    protected function table(array $headings, array $values, Log $log = Log::NONE): void {
        $maxLen = [];
        foreach ($headings as $heading) {
            $maxLen[$heading] = strlen($heading);
        }
        foreach ($values as $key => $value) {
            if (is_array($value)) {
                foreach ($value as $k => $v) {
                    $this->setMaxLen($maxLen, $k, $v);
                }
            } else {
                $this->setMaxLen($maxLen, $key, $value);
            }
        }

        $str = '';

        foreach ($headings as $heading) {
            $str .= "| {$heading} ";
            if (strlen($heading) < $maxLen[$heading]) {
                $str .= $this->repeatChar(' ', $maxLen[$heading] - strlen($heading));
            }
        }
        $str .= "|\n";

        $str .= "|";
        foreach ($headings as $heading) {
            $str .= $this->repeatChar('-', $maxLen[$heading] + 2);
            $str .= '+';
        }
        $str = substr_replace($str, "|\n", -1);

        $multi = false;
        foreach ($values as $key => $value) {
            if (is_array($value)) {
                $multi = true;
                foreach ($value as $k => $v) {
                    $str .= "| {$v} ";
                    $str .= $this->repeatChar(' ', $maxLen[$k] - strlen($v));
                }
                $str .= "|\n";
            } else {
                $str .= "| {$value} ";
                $str .= $this->repeatChar(' ', $maxLen[$key] - strlen($value));
            }
        }
        if (!$multi) {
            $str .= "|\n";
        }
        $this->writeln($str, $log);
    }

    /**
     * In debug mode, print the data to the screen.
     */
    protected function printProtoFile(\Google\Protobuf\Internal\Message $bin, Log $log = Log::DEBUG): void {
        $this->writeln($bin->serializeToJsonString() . PHP_EOL, $log);
        $this->writeln(bin2hex($bin->serializeToString()) . PHP_EOL, $log);
    }
}
