<?php

namespace onespace\tools\enum\oneaccess;

use onespace\tools\enum\base\UnbackedHelper;

enum AccessEventCode {
    use UnbackedHelper;

    case ACCESS_ALLOWED;
    case ACCESS_CONFIRMED;
    case ACCESS_UNCONFIRMED;

    case ACCESS_METHOD_DENIED_UNKNOWN;
    case ACCESS_METHOD_DENIED_ZERO_UID_EVENT;
    case ACCESS_METHOD_DENIED_DAYS_OF_WEEK;
    case ACCESS_METHOD_DENIED_NO_RIGHTS;
    case ACCESS_METHOD_DENIED_USER_NOT_FOUND;
    case ACCESS_METHOD_DENIED_GATEWAY_DISABLED;
    case ACCESS_METHOD_DENIED_READER_UNKNOWN;

    case ACCESS_RIGHTS_DENIED_READER;
    case ACCESS_RIGHTS_DENIED_DAYS_OF_WEEK;
    case ACCESS_RIGHTS_DENIED_TIME_OF_DAY;

    case ACCESS_USER_DENIED_NO_RIGHTS;
    case ACCESS_USER_DENIED_NO_ENTRIES_LEFT;
    case ACCESS_USER_DENIED_NO_EXITS_LEFT;
    case ACCESS_USER_DENIED_FUTURE_DATED;
    case ACCESS_USER_DENIED_EXPIRED;
    case ACCESS_USER_DENIED_ENTRY_ALL_EXITS_REDEEMED;
    case ACCESS_USER_DENIED_EXIT_ALL_ENTRIES_REDEEMED;

    private function config(string $key): string|int {
        $config = match ($this) {
            self::ACCESS_ALLOWED => [
                'id' => 1,
                'name' => 'Access Allowed',
                'description' => 'An Access Method, Rights and optionally a User met all criteria to access a place',
            ],
            self::ACCESS_CONFIRMED => [
                'id' => 2,
                'name' => 'Access Confirmed',
                'description' => 'An Input activated event occurred which suggests the access gateway successfully opened the door/gate',
            ],
            self::ACCESS_UNCONFIRMED => [
                'id' => 3,
                'name' => 'Access Unconfirmed',
                'description' => 'A relay deactivated event occured before the Input Activated Event with Access Confirmation',
            ],
            self::ACCESS_METHOD_DENIED_UNKNOWN => [
                'id' => 4,
                'name' => 'Access Method Denied Unknown',
                'description' => 'The method could not be found in the database',
            ],
            self::ACCESS_METHOD_DENIED_ZERO_UID_EVENT => [
                'id' => 5,
                'name' => 'Access Method Denied Zero UID Event',
                'description' => 'The reader reported zero for this method, which is not allowed',
            ],
            self::ACCESS_METHOD_DENIED_DAYS_OF_WEEK => [
                'id' => 6,
                'name' => 'Access Method Denied Days Of Week',
                'description' => 'The method was denied based on its DoW configured',
            ],
            self::ACCESS_METHOD_DENIED_NO_RIGHTS => [
                'id' => 7,
                'name' => 'Access Method Denied No Rights',
                'description' => 'The method has no access rights configured',
            ],
            self::ACCESS_METHOD_DENIED_USER_NOT_FOUND => [
                'id' => 8,
                'name' => 'Access Method Denied User Not Found',
                'description' => 'The associated user with this method was not found',
            ],
            self::ACCESS_METHOD_DENIED_GATEWAY_DISABLED => [
                'id' => 9,
                'name' => 'Access Method Denied Gateway Disabled',
                'description' => 'Gateway is disabled',
            ],
            self::ACCESS_METHOD_DENIED_READER_UNKNOWN => [
                'id' => 10,
                'name' => 'Access Method Denied Reader Unknown',
                'description' => 'Reader is unknown or not configured',
            ],
            self::ACCESS_RIGHTS_DENIED_READER => [
                'id' => 11,
                'name' => 'Access Rights Denied Reader',
                'description' => 'User not allowed access on this reader',
            ],
            self::ACCESS_RIGHTS_DENIED_DAYS_OF_WEEK => [
                'id' => 12,
                'name' => 'Access Rights Denied Days Of Week',
                'description' => 'Access was denied based on the DoW Rights configured',
            ],
            self::ACCESS_RIGHTS_DENIED_TIME_OF_DAY => [
                'id' => 13,
                'name' => 'Access Rights Denied Time Of Day',
                'description' => 'Access was denied based on the ToD Rights configured',
            ],
            self::ACCESS_USER_DENIED_NO_RIGHTS => [
                'id' => 14,
                'name' => 'Access User Denied No Rights',
                'description' => 'User does not have any access rights configured',
            ],
            self::ACCESS_USER_DENIED_NO_ENTRIES_LEFT => [
                'id' => 15,
                'name' => 'Access User Denied No Entries Left',
                'description' => 'The visitor has no entries left',
            ],
            self::ACCESS_USER_DENIED_NO_EXITS_LEFT => [
                'id' => 16,
                'name' => 'Access User Denied No Exits Left',
                'description' => 'The visitor has no exits left',
            ],
            self::ACCESS_USER_DENIED_FUTURE_DATED => [
                'id' => 17,
                'name' => 'Access User Denied Future Dated',
                'description' => 'The visitor does not have access yet',
            ],
            self::ACCESS_USER_DENIED_EXPIRED => [
                'id' => 18,
                'name' => 'Access User Denied Expired',
                'description' => 'The visitor no longer has access past the date and time',
            ],
            self::ACCESS_USER_DENIED_ENTRY_ALL_EXITS_REDEEMED => [
                'id' => 19,
                'name' => 'Access User Denied Entry, All Exits Redeemed',
                'description' => 'The visitor has used all their exits and tried to enter',
            ],
            self::ACCESS_USER_DENIED_EXIT_ALL_ENTRIES_REDEEMED => [
                'id' => 20,
                'name' => 'Access User Denied Exit, All Entries Redeemed',
                'description' => 'The visitor has used all their entries and tried to exit, with FirstExitThenEnters enabled',
            ],
        };
        return $config[$key];
    }

    public function id(): int {
        return $this->config('id');
    }

    public function name(): string {
        return $this->config('name');
    }

    public function description(): string {
        return $this->config('description');
    }
}
