<?php

namespace onespace\tools\activeApi\models\oneaccess;

use onespace\tools\activeApi\components\clients\OneAccessClient;
use thamtech\uuid\helpers\UuidHelper;
use yii\helpers\ArrayHelper;
use yii\web\NotFoundHttpException;

/**
 * This class is intended to be extended from within the child service,
 * though it shouldn't need to be if you just need * a basic interface
 * with the method / endpoint without any extra methods.
 *
 * @property integer $id
 * @property string $guid
 * @property string $site_id
 * @property string $name
 * @property string $visit_id
 * @property string $OSID
 * @property integer $entry_uses
 * @property integer $exit_uses
 * @property integer $entries_redeemed
 * @property integer $exits_redeemed
 * @property string $start_datetime
 * @property string $end_datetime
 * @property integer $exit_rollover_enabled
 * @property string $date_created
 * @property string $date_updated
 *
 * @author  Gareth Palmer <gareth@one-space.co.za>
 */

class AccessUser extends \onespace\tools\activeApi\models\BaseActiveApiModel {
    public $id;
    public $guid;
    public $site_id;
    public $name;
    public $visit_id;
    public $OSID;
    public $entry_uses;
    public $exit_uses;
    public $entries_redeemed;
    public $exits_redeemed;
    public $start_datetime;
    public $end_datetime;
    public $exit_rollover_enabled;
    public $date_created;
    public $date_updated;

    public array $extraDetails = [];

    public const VMS4_VISITOR_SCENARIO_DEFAULT  = null;
    public const VMS4_VISITOR_SCENARIO_ADMIN    = 0b01;
    public const VMS4_VISITOR_SCENARIO_GATEBOOK = 0b10;

    /**
     * {@inheritdoc}
     */
    public function rules() {
        return ArrayHelper::merge(parent::rules(), [
            [['site_id', 'start_datetime', 'end_datetime'], 'required'],
            [['entry_uses', 'exit_uses', 'entries_redeemed', 'exits_redeemed', 'exit_rollover_enabled'], 'integer'],
            [['start_datetime', 'end_datetime'], 'safe'],
            [['guid', 'site_id', 'visit_id', 'OSID'], 'string', 'max' => 36],
            [['name'], 'string', 'max' => 255],
            // [['guid'], 'unique']
        ]);
    }

    /**
     * {@inheritdoc}
     */
    public function attributeLabels() {
        return ArrayHelper::merge(parent::attributeLabels(), [
            'id' => 'ID',
            'guid' => 'Guid',
            'site_id' => 'Site ID',
            'name' => 'Name',
            'visit_id' => 'Visit ID',
            'OSID' => 'Osid',
            'entry_uses' => 'Entry Uses',
            'exit_uses' => 'Exit Uses',
            'entries_redeemed' => 'Entries Redeemed',
            'exits_redeemed' => 'Exits Redeemed',
            'start_datetime' => 'Start Datetime',
            'end_datetime' => 'End Datetime',
            'exit_rollover_enabled' => 'Exit Rollover Enabled',
        ]);
    }

    /**
     * {@inheritdoc}
     */
    public function attributeHints() {
        return ArrayHelper::merge(parent::attributeHints(), [
            'id' => 'Primary Key',
            'guid' => 'GUID of this gateway - mapped to Property service',
            'site_id' => 'Site ID',
            'name' => 'Name of the user if known',
            'visit_id' => 'ID of the visit that created this, if applicable',
            'OSID' => 'OSID of the user, if applicable',
            'entry_uses' => 'Number of times this user is allowed to enter the site',
            'exit_uses' => 'Number of times this user is allowed to exit the site',
            'entries_redeemed' => 'Number of times this user has actually entered the site',
            'exits_redeemed' => 'Number of times this user has actually exited the site',
            'start_datetime' => 'Date and time user should gain access',
            'end_datetime' => 'Date and time user should lose access',
            'exit_rollover_enabled' => 'Whether this user should be able to exit on the following day or not',
        ]);
    }

    /**
     * {@inheritdoc}
     */
    public function client(): OneAccessClient {
        return new OneAccessClient();
    }

    /**
     * {@inheritdoc}
     */
    public function primaryKey(): string {
        return 'id';
    }

    /**
     * {@inheritdoc}
     */
    public function baseEndpoints(): array {
        return [
            'v1' => 'v1/access-user',
            'v2' => 'v2/access-user',
        ];
    }

    public function beforeSave(bool $insert): bool {
        if (parent::beforeSave($insert)) {
            $this->date_updated = date('Y-m-d H:i:s');
            if ($insert) {
                $this->guid = UuidHelper::uuid();
                $this->entries_redeemed = 0;
                $this->exits_redeemed = 0;
                $this->date_created = date('Y-m-d H:i:s');
            }
            return true;
        }
        return false;
    }

    /**
     * @throws NotFoundHttpException
     */
    public static function findByGuid(string $guid): static {
        $model = self::findOne(['guid' => $guid]);
        if ($model === null) {
            throw new NotFoundHttpException('Access User not found');
        }

        return $model;
    }
}
