<?php

namespace onespace\tools\activeApi\components\clients;

use onespace\tools\activeApi\components\clients\base\BaseOneSpaceClient;
use Yii;

/**
 * The client to connect to the Social Messenger Microservice
 *
 * Extend from here to add functionality.
 *
 * Add the following to `config/params.php`:
 *
 * ```php
 * 'serviceUrls' => [
 *  ...
 *  'socialMessenger' => getenv('SERVICE_URL_SOCIAL_MESSENGER') ?: 'https://dev.social-messenger.1s.co.za/',
 *  ...
 * ],
 * ```
 */

class SocialMessengerClient extends BaseOneSpaceClient {
    public function serverUrl(): string {
        return Yii::$app->params['serviceUrls']['socialMessenger'];
    }

    /**
     * Join the bot to a group, given an invite link or code
     * @param string $botCellNumber Cell number of the bot to add
     * @param string $invite Invite code or link for the group
     * @return bool
     */
    public function joinGroup(string $botCellNumber, string $invite): bool {
        $url = $this->serverUrl() . 'v1/group/join';
        Yii::info($url, __METHOD__);
        $request = $this->getClient()->post($url, [
            'data' => [
                'providerCell' => $botCellNumber,
                'invite' => $invite,
            ],
        ]);
        $this->setLastRequest($request);
        $response = $request->send();
        if ($response->isOk) {
            return $response->data['success'];
        }
        return false;
    }

    /**
     * Add a new user to a group
     * @param string $cellNumber The phone number of the person to add
     * @param string $groupId ID of the group to add the person to
     * @return bool
     */
    public function addUserToGroup(string $cellNumber, string $groupId): bool {
        $url = $this->serverUrl() . 'v1/group/add';
        Yii::info($url, __METHOD__);
        $request = $this->getClient()->post($url, [
            'data' => [
                'cell' => $cellNumber,
                'groupId' => $groupId,
            ],
        ]);
        $this->setLastRequest($request);
        $response = $request->send();
        if ($response->isOk) {
            return $response->data['success'];
        }
        return false;
    }

    /**
     * Remove a user from a group
     * @param string $cellNumber Cell number of the person to remove
     * @param string $groupId ID of the group to remove the person from
     * @return bool
     */
    public function removeUserFromGroup(string $cellNumber, string $groupId): bool {
        $url = $this->serverUrl() . 'v1/group/remove';
        Yii::info($url, __METHOD__);
        $request = $this->getClient()->post($url, [
            'data' => [
                'cell' => $cellNumber,
                'groupId' => $groupId,
            ],
        ]);
        $this->setLastRequest($request);
        $response = $request->send();
        if ($response->isOk) {
            return $response->data['success'];
        }
        return false;
    }

    /**
     * Get the current confirmation code for a given site
     * @param string $siteSerial GUID of the site to look for
     * @return array
     */
    public function getSiteConfirmCode(string $siteSerial): array {
        $url = $this->serverUrl() . 'v1/site/code' . '?' . http_build_query(['site' => $siteSerial]);
        $request = $this->getClient()->get($url);
        $this->setLastRequest($request);
        $response = $request->send();
        if (!$response->isOk) {
            throw new NotFoundHttpException("Cannot get site code");
        }
        return $response->data['data'];
    }
}
