<?php

namespace onespace\tools\activeApi\components\clients\teams\text;

use Exception;
use onespace\tools\activeApi\components\clients\teams\TeamsMessageComponent;

/**
 * A General text block within a teams message
 *
 * @see https://adaptivecards.io/schemas/adaptive-card.json
 * @author Gareth Palmer <gareth@one-space.co.za>
 */
final class TeamsTextBlock implements TeamsMessageComponent {
    private const string TYPE = "TextBlock";

    public function __construct(
        /**
         * Text to display. A subset of markdown is supported (https://aka.ms/ACTextFeatures)
         */
        public string $text,
        /** Controls the color of `TextBlock` elements. */
        public TextColour $color = TextColour::UNSET,
        /** Controls size of text. */
        public TextSize $size = TextSize::UNSET,
        /** Controls the weight of `TextBlock` elements. */
        public TextWeight $weight = TextWeight::UNSET,
        /** Type of font to use for rendering */
        public TextFontType $fontType = TextFontType::UNSET,
        /**
         * Controls the horizontal text alignment. When not specified, the value
         * of horizontalAlignment is inherited from the parent container. If no
         * parent container has horizontalAlignment set, it defaults to Left.
         */
        public TextHorizontalAlignment $horizontalAlignment = TextHorizontalAlignment::UNSET,
        /** Controls how a TextBlock behaves. */
        public TextStyle $style = TextStyle::UNSET,
        /** "If `true`, allow text to wrap. Otherwise, text is clipped. */
        public bool $wrap = true,
        /** If `true`, displays text slightly toned down to appear less prominent. */
        public ?bool $isSubtle = null,
    ) {
    }

    #[\Override]
    public function build(): array {
        if (!isset($this->text)) {
            throw new Exception("Text not set");
        }
        $payload = [
            'type' => self::TYPE,
            'text' => $this->text,
            'wrap' => $this->wrap,
        ];

        if ($this->color !== TextColour::UNSET) {
            $payload['color'] = $this->color->value;
        }
        if ($this->size !== TextSize::UNSET) {
            $payload['size'] = $this->size->value;
        }
        if ($this->weight !== TextWeight::UNSET) {
            $payload['weight'] = $this->weight->value;
        }
        if ($this->fontType !== TextFontType::UNSET) {
            $payload['fontType'] = $this->fontType->value;
        }
        if ($this->horizontalAlignment !== TextHorizontalAlignment::UNSET) {
            $payload['horizontalAlignment'] = $this->horizontalAlignment->value;
        }
        if ($this->style !== TextStyle::UNSET) {
            $payload['style'] = $this->style->value;
        }
        if (!is_null($this->isSubtle)) {
            $payload['isSubtle'] = $this->isSubtle;
        }

        return $payload;
    }
}
