<?php

namespace onespace\tools\activeApi\models\auth;

use \onespace\tools\activeApi\components\clients\OneSpaceIdClient;
use \onespace\tools\activeApi\models\enum\NotificationMethod;
use \onespace\tools\activeApi\models\enum\NotificationAction;
use Yii;
use yii\base\Model as BaseModel;

class Notification extends BaseModel {

    /**
     * The user to be notified. Form of GUID.
     */
    public string $target;

    /**
     * Category of notification.
     * Lower snake case
     * e.g access_event
     */
    public string $category;

    /**
     * Notification subject.
     */
    public string $title;

    /**
     * Short description of notification.
     */
    public string $summary;

    /**
     * Notification message.
     */
    public string $message;

    /**
     * Methods to send the notification via.
     * @see NotificationMethod Enum
     */
    public ?array $methods = null;

    /**
     * Page in Connect to redirect to.
     * Defaults to show messages
     * @see NotificationAction Enum
     */
    public ?string $pushAction = null;

    /**
     * Payload to pass to notification. e.g. visit guid.
     */
    public ?string $pushPayload = null;

    /**
     * Click here newsletter link for email body.
     */
    public ?string $viewOnlineUrl = null;

    public function __construct($config = []) {
        parent::__construct($config);

        if (!$this->pushAction) {
            $this->pushAction = NotificationAction::NOTIFICATION_ACTION_SHOW_MESSAGES->value;
        }

        if (!$this->methods) {
            $methods = function (): array {
                if (YII_ENV_DEV) {
                    return [
                        NotificationMethod::METHOD_EMAIL->value,
                    ];
                } else {
                    return [
                        NotificationMethod::METHOD_PUSH_NOTIFICATION->value,
                        // NotificationMethod::METHOD_SMS->value
                    ];
                }
            };
            $this->methods = $methods();
        }

        if (!$this->pushPayload) {
            $this->pushPayload = "";
        }
    }

    public function rules() {
        return [
            'required' => [['target', 'category', 'title', 'summary', 'message',], 'required'],
            'optional' => [['pushPayload'], 'string'],
        ];
    }

    /**
     * Sends a request to the OneSpaceIdClient to notify a user.
     * @see OneSpaceIdClient
     */
    public function send(): bool {

        Yii::debug("Sending user notification: " . json_encode($this->toArray()), __METHOD__);

        $payload = [
            'target' => $this->target,
            'category' => $this->category,
            'title' => $this->title,
            'summary' => $this->summary,
            'message' => $this->message,
            'methods' => $this->methods,
            'pushAction' => $this->pushAction,
            'pushPayload' => $this->pushPayload,
            'viewOnlineUrl' => $this->viewOnlineUrl,
        ];

        $client = new OneSpaceIdClient();
        return $client->notifyUser($payload);
    }
}
